#include <iostream>
#include <sys/stat.h>
#include "constants.h"
#include "configuration.h"

Configuration::Configuration()
{
	setDefaults();
}

/** Sets all settings to default values.
  */
void Configuration::setDefaults()
{
	hudType = DEFAULT_HUD_TYPE;
	numStars = DEFAULT_NUM_STARS;
	fullscreen = DEFAULT_FULLSCREEN;
	width = DEFAULT_WIDTH;
	height = DEFAULT_HEIGHT;
}

/** @brief Loads the configuration settings from the config file.
  *
  * If that file does not exist settings will be set to defaults.
  * If a specific option line is not found in the file the setting also will
  * be set to its default value.
  */
void Configuration::load()
{
	char	filename[256];
	char	option[256];
	int		value;

	setDefaults();
	getFileName(filename);
	FILE* pFile = fopen(filename, "rb"); // no CR/LF translation
	if ( pFile == NULL )
		return;

	do
	{
		fscanf(pFile, "%s = %d", option, &value);
		if ( strcmp(option, CONFIG_HUDTYPE)==0 )
			hudType = (HUD_TYPE)value;
		else if ( strcmp(option, CONFIG_NUMSTARS) == 0 )
			numStars = value;
		else if ( strcmp(option, CONFIG_FULLSCREEN) == 0 )
			fullscreen = value==0 ? false : true;
		else if ( strcmp(option, CONFIG_WIDTH) == 0 )
			width = value;
		else if ( strcmp(option, CONFIG_HEIGHT) == 0 )
			height = value;
	} while ( !feof(pFile) );
	fclose(pFile);
}

/** @brief Saves the configuration settings to the config file.
  */
void Configuration::save()
{
	char filename[256];

#ifndef _MSC_VER
	char pathname[256];
	struct stat dummy;
	sprintf(pathname, "%s/.%s", getenv("HOME"), EXECUTABLE);
	if ( stat(pathname, &dummy) == -1 )
		mkdir(pathname, 0777);
#endif
	getFileName(filename);
	FILE* pFile = fopen(filename, "w");
	if ( pFile == NULL )
	{
		std::cerr << "error while trying to create config file" << std::endl;
		return;
	}
	fprintf(pFile, "%s = %d\n", CONFIG_HUDTYPE, hudType);
	fprintf(pFile, "%s = %d\n", CONFIG_NUMSTARS, numStars);
	fprintf(pFile, "%s = %d\n", CONFIG_FULLSCREEN, fullscreen ? 1 : 0 );
	fprintf(pFile, "%s = %d\n", CONFIG_WIDTH, width);
	fprintf(pFile, "%s = %d\n", CONFIG_HEIGHT, height);
	fclose(pFile);
}

/** @brief Creates the config file name.
  *
  * For Win32, the config file will always be put into the current working
  * directory. All other systems (assumed to be *nix-like) will have the file
  * in $HOME/.netwarsgl/nwconfig.
  */
void Configuration::getFileName(char* filename)
{
#ifdef _MSC_VER
	sprintf(filename, "%s", CONFIG_FILENAME); // Win32: always use current path
#else
	sprintf(filename, "%s/.%s/%s", getenv("HOME"), EXECUTABLE, CONFIG_FILENAME);
#endif
}

