#include <GL/glut.h>
#include <math.h>
#include <cstdio>
#include <string>
#include <cstdlib>
#include "stdinclu.h"
#include "constants.h"
#include "vector3d.h"
#include "bullet.h"
#include "missile.h"
#include "spaceobject.h"
#include "particle.h"
#include "particlesystem.h"
#include "level.h"
#include "game.h"
#include "pirateai.h"
#include "commanderai.h"
#include "text.h"
#include "timing.h"
#include "configuration.h"
//#include "nwsound.h"

// external global variables
extern int				giWidth, giHeight, giWidthHalf, giHeightHalf;
extern Configuration	config;
//extern NWSound*			pSound;

// external functions
void error(const char *msg);

// game specific constants
const char  szFirePower[] =	"EXTRA FIRE POWER",
						szCloak[] =		"CLOAK",
						szThrust[] =	"EXTRA THRUST",
						szMissiles[] =	"MISSILES: %d",
						szScore[] =		"SCORE: %d",
						szMsslWarn[] =	"WARNING: MISSILE LAUNCH DETECTED",
						szPodWarn[] =	"WARNING: POD IN DANGER",
						szPodDead[] =	"POD DESTROYED";

//constructor
Game::Game()
{
	initialize();
}

Game::~Game()
{
	// TODO delete allocated member stuff
	shutdown();
}

/** @brief Initializes text-related variables.
  */
void Game::setupTextSystem()
{
	iTextWidth = glutBitmapLength(GLUT_BITMAP_8_BY_13, (unsigned char*)szFirePower);
	iTextHeight = 14;
}

/** @brief Adds a power-up.
  *
  * Power-ups (called satellites in NetWars) are objects that just hang around
  * and wait to be collected. Note that in this implementation, the pods are
  * also treated as power-ups.
  *
  * @param type The type of power-up to be added. If it is a pod, the AI of
  *             the object set accordingly.
  * @param x The x coordinate of the power-up's position.
  * @param y The y coordinate of the power-up's position.
  * @param z The z coordinate of the power-up's position.
  */
void Game::addPowerup(PowerUpType type, GLdouble x, GLdouble y, GLdouble z)
{
	int i=0;

	while (ppowerups[i] && i<MAX_POWERUPS)
		i++;
	
	if (i==MAX_POWERUPS)
		return;

	int listno = 0;
	switch (type)
	{
		case HEALTH: listno = DL_HEALTH; break;
		case FIREPOWER: listno = DL_FIREPOWER; break;
		case MISSILES: listno = DL_XTRAMSSL; break;
		case THRUST: listno = DL_THRUST; break;
		case CLOAK: listno = CLOAK; break;
		case POD: listno = DL_POD; break;
		case POWERUP_TYPE_COUNT: /* here to avoid default */ break;
	}
	ppowerups[i] = new SpaceObject(listno, x, y, z);
	ppowerups[i]->setAutoRotation(0.0, 0.18f, 0.0);
	if ( type == POD )
	{
		ppowerups[i]->setAI(POD_AI);
		podcount++;
	}
}

/** @brief Adds an AI controlled ship.
  *
  * Only a specific number of AI ships are allowed to be in the game space
  * at once.
  * When adding a new ship, its AI is set according to the AI type.
  *
  * @param type The type of AI ship to be added.
  * @param x The x coordinate of the ship's position.
  * @param y The y coordinate of the ship's position.
  * @param z The z coordinate of the ship's position.
  */
void Game::addAIShip(AIShipType type, GLdouble x, GLdouble y, GLdouble z)
{
	int i=0;

	while (pships[i] && i<MAX_SHIPS)
		i++;
	
	if (i==MAX_SHIPS)
		return;

	int listno = 0, interval=0;
	switch (type)
	{
		case NO_AI: break;
		case PIRATE: listno = DL_PIRATE; break;
		case FIGHTER: listno = DL_FIGHTER; interval=FIGHTER_BULLET_INTERVAL; break;
		case GUARDIAN: listno = DL_GUARDIAN; interval=GUARDIAN_BULLET_INTERVAL; break;
		case COMMANDER: listno = DL_COMMANDER; interval=COMMANDER_BULLET_INTERVAL; break;
		case POD_AI: listno = DL_POD; break; // won't be used anyway
		case AI_TYPE_COUNT: break;
	}
	pships[i] = new SpaceObject(listno, x, y, z);
	pships[i]->setAI(type);
	pships[i]->setBulletInterval(interval);
	if ( type == PIRATE )
	{
		((PirateAI*)(pships[i]->getAI()))->setInitialTarget( PLAYER );
		((PirateAI*)(pships[i]->getAI()))->setGame( this );
	}
	else if ( type == COMMANDER )
		((CommanderAI*)(pships[i]->getAI()))->setInitialTargets( PLAYER, getFreePod() );
	else
		(pships[i]->getAI())->setTarget(PLAYER);

	aishipcount++;
}

/** @brief Starts a new game with the given level.
  *
  * This method should be called each time a new game is started.
  * It performs per-game initializations, then per-level initializations.
  * @param pstartlvl The level being played first.
  */
void Game::startNewGame(Level* pstartlvl)
{
	score = 0;
	framecount = 0;
	startcount = oldtickcount = getElapsedTime()-1;	// avoid division by zero
	gameover_int = false;
	gameover_ext = false;
	addPowerup(POD,  PODPOS, 0,  PODPOS);
	addPowerup(POD, -PODPOS, 0,  PODPOS);
	addPowerup(POD,  PODPOS, 0, -PODPOS);
	addPowerup(POD, -PODPOS, 0, -PODPOS);
	addPowerup(POD, 0, PODPOS, 0);
	addPowerup(POD, 0, -PODPOS, 0);

	startNewLevel(pstartlvl);
}

/** @brief Starts a new level with the current game.
  *
  * This method resets all required members to a fresh level start state.
  * It is here to have a fast way of switching levels without destroying
  * and re-creating a Game object for every level.
  *
  * @param plvl The level to be played next.
  */
void Game::startNewLevel(Level* plvl)
{
	lastpowerupspawned = 0;
	bForwardPressed = bBackwardPressed = lbuttonDown = rbuttonDown = false;
	PLAYER->setPosition(0, 0, 0);
	PLAYER->setVelocity(0);
	levelover_int = false;
	levelover_ext = false;
	setMessage("");
	aishipcount = 0;
	//activePartSysCount = 0;

	plevel = plvl;
	plevel->start(this);

	addPowerupEvent(8000);
}

/** @brief Members initialization.
  *
  * This method performs one-time initializations for data members.
  */
void Game::initialize()
{
	setupTextSystem();
	fRadcentx = giWidthHalf;
	fRadcenty = giHeight * 0.12;
	fRadarZoom = 0.000005;
	bShowFPS = false;
	podcount = 0;

	int i;
	for ( i=0; i < MAX_SHIPS; i++ )
		pships[i] = NULL;
	for ( i=0; i < MAX_POWERUPS; i++ )
		ppowerups[i] = NULL;
	
	PLAYER = new SpaceObject(0);
	PLAYER->setBulletInterval(PLAYER_BULLET_INTERVAL);
}

/** @brief Start trigger.
  *
  * Initializes the tick counter with the current value to avoid jumping
  * objects within the first few frames. Therefore, this method should
  * be called directly before GLUT control is handed over here.
  */
void Game::start()
{
	oldtickcount = getElapsedTime();
}

/** @brief Stop trigger.
  *
  * Stops all missiles.
  * @todo Do we need this?
  */
void Game::stop()
{
	missiles.clear();
}

/**
  * Returns a pointer to a free pod.
  * A pod is called free if it is not currently being hijacked or returning
  * to its home position.
  */
SpaceObject* Game::getFreePod()
{
	//int podno = 0;
	PodAI *pAI;

	for ( int i=0; ppowerups[i] && i<MAX_POWERUPS; i++ )
		if ( ppowerups[i]->getModel() == DL_POD )
		{
			pAI = (PodAI*)(ppowerups[i]->getAI());
			if ( !(pAI->isBeingHijacked()) )
				return ppowerups[i];
		}

	return NULL;
}

void Game::reshapeFunc()
{
	fRadcentx = giWidthHalf;
	fRadcenty = giHeight * 0.12;
	anscx = giWidth * 0.28;
	anscy = giHeight * 0.02;
	answidth = giWidth * 0.1;
	ansheight = giHeight * 0.18;
	ansthrx = giWidth * 0.09;
}

void Game::shutdown()
{
	int i;

	for ( i=0; i<MAX_POWERUPS; i++ )
		if ( ppowerups[i] )
			delete ppowerups[i];
	for ( i=0; i<MAX_SHIPS; i++ )
		if ( pships[i] )
			delete pships[i];
}

/** @brief Sets the message currently being displayed.
  *
  * Messages are displayed in the upper part of the HUD. By calling this
  * method, not only the message to be displayed is set, but also the
  * 'message timer' gets (re-)initialized. This allows the Game object
  * to display the message for a specific duration only and then take it away
  * without an extra method call.
  *
  * @param msg The message to be displayed.
  */
void Game::setMessage(const char* msg)
{
	message = (char*)msg;
	lastmessagetickcount = getElapsedTime();
	fMessageX = (giWidth-glutBitmapLength(GLUT_BITMAP_8_BY_13, (unsigned char*)msg))*0.5;
}

/** @brief Draws a line into the radar grid.
  *
  * This method does not modify the OpenGL state (that is, the line is drawn by
  * glVertex calls, using the current line settings and the current color).
  * Further, the line coordinates are calculated to be used with a 2d (ortho)
  * projection matrix.
  *
  * @param v The position that the radar line should indicate, game space
  *          absolute (not relative to the player).
  *
  * @todo Maybe this could be achieved with OpenGL, which would be both more
  * elegant and efficient since now there is too much math in here. Thoughts
  * are: OpenGL viewport, use a 3D projection matrix and then use the object's
  * coordinates directly.
  */
void Game::drawRadarLine(const Vector3D& v)
{
	Vector3D		pos0=PLAYER->getPosition(),
					deltaPos = pos0 - v, base;
	GLfloat			d,
					basex, basey,
					endx, endy;
	short			dsign=1;

	// distance of object to player's lookdir-right-plane
	d = 0.01*DotProduct(PLAYER->getUp(), deltaPos);
	if ( d<0 )
		dsign = -1;	// the sign will be needed more often
	base = deltaPos + PLAYER->getUp()*(-d);
	basex = fRadcentx - giWidth*fRadarZoom*DotProduct(base, PLAYER->getRight());
	basey = fRadcenty - giWidth*fRadarZoom*DotProduct(base, PLAYER->getLookDir());
	endx = basex;
	if ( fabs(d)<3 )
		d = 3 * dsign;
	endy = basey - d;
	// draw the primary line
	glVertex2f(basex, basey);
	glVertex2f(endx, endy);
	// draw the crossing line to make it look like a little arrow
	glVertex2f(endx-1, endy + dsign*2);
	glVertex2f(endx+2, endy + dsign*2);
}

/** @brief Draws the HUD, that is, all non-3D objects on the screen.
  *
  * This includes text, the radar, shield and velocity indicators and power-up
  * states.
  *
  * The different HUD types are handled by a simple 'if' only.
  *
  * @todo Cut this awfully big method into smaller pieces.
  * @todo More flexible HUD type management.
  */
void Game::drawHUD()
{
	if ( gameover_int )
	{
		glDisable(GL_LIGHTING);
		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
			glLoadIdentity();
			gluOrtho2D(0, giWidth, 0, giHeight);
			glColor3d(GLRED);
			fMessageX = (giWidth-glutBitmapLength(GLUT_BITMAP_8_BY_13, (const unsigned char*)GAMEOVER_TEXT))*0.5;
			Text::printCenteredAt(giWidthHalf, MESSAGE_YCOORD, GAMEOVER_TEXT);
		glPopMatrix();
		glMatrixMode(GL_MODELVIEW);
		glEnable(GL_LIGHTING);
	}
	else
	{
		int			vel = PLAYER->getVelocityPercent(),
					i,
					textxpos = giWidth-iTextWidth;
		GLfloat		velBegin = vel>0 ? giHeight*0.212 : giHeight*0.208,
					velHeight = giHeight*0.19 * vel*0.01,
					cursorwidth = giWidth*0.006,
					extra; // this was an int before
		Vector3D	pos, deltaPos, v,
					pos0 = PLAYER->getPosition(), 
					center = PLAYER->getPosition() + PLAYER->getLookDir(),
					base, end;
		char		str[256];
		long		tickcount = getElapsedTime();
		bool		locked = false;

		if ( framecount % 20 == 0 )
		{
			oldmx = mousex;
			oldmy = mousey;
		}

		glDisable(GL_LIGHTING);
		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
			glLoadIdentity();
			gluOrtho2D(0, giWidth, 0, giHeight);
			glColor3d(GLRED);
			glBegin(GL_LINES);
				// mouse cursor diagonal lines
				glVertex2f(oldmx-giWidth*0.05, oldmy-giWidth*0.04); glVertex2f(oldmx-giWidth*0.065, oldmy-giWidth*0.025);
				glVertex2f(oldmx-giWidth*0.05, oldmy+giWidth*0.04); glVertex2f(oldmx-giWidth*0.065, oldmy+giWidth*0.025);
				glVertex2f(oldmx+giWidth*0.05, oldmy-giWidth*0.04); glVertex2f(oldmx+giWidth*0.065, oldmy-giWidth*0.025);
				glVertex2f(oldmx+giWidth*0.05, oldmy+giWidth*0.04); glVertex2f(oldmx+giWidth*0.065, oldmy+giWidth*0.025);
			glEnd();
			glColor3d(GLWHITE);
			////////// cursor //////////////
			glBegin(GL_LINE_STRIP);
				glVertex2f(mousex-cursorwidth, mousey-cursorwidth);
				glVertex2f(mousex-cursorwidth, mousey+cursorwidth);
				glVertex2f(mousex+cursorwidth, mousey+cursorwidth);
				glVertex2f(mousex+cursorwidth, mousey-cursorwidth);
				glVertex2f(mousex-cursorwidth, mousey-cursorwidth);
			glEnd();
			////////////// radar ////////////////
			glBegin(GL_LINES);
				/////// power-ups
				for ( i=0; ppowerups[i]!=0 && i<MAX_POWERUPS; i++ )
				{
					switch ( ppowerups[i]->getModel() )
					{
						case DL_POD: glColor3d(GLMAGENTA); break;
						default: glColor3d(GLWHITE); break;
					}
					drawRadarLine(ppowerups[i]->getPosition());
				}
				/////// ships
				glColor3d(GLGREEN);
				for ( i=1; pships[i]!=NULL && i<MAX_SHIPS; i++ )
					drawRadarLine(pships[i]->getPosition());
				// missiles
				glColor3d(GLYELLOW);
				for ( size_t m=0; m < missiles.size(); ++m )
					drawRadarLine(missiles[m].getPosition());
				// radar grid
				glColor3d(GLDARK_RED);
				for ( i=-2; i<=2; i++ )
				{
					glVertex2f(fRadcentx + i*giWidth*0.06,   fRadcenty - 2*giWidth*0.025);
					glVertex2f(fRadcentx + i*giWidth*0.04, 1+fRadcenty + 2*giWidth*0.025);
					glVertex2f(  fRadcentx - (2*giWidth*(0.06+(0.005*(-i-2)))), fRadcenty + i*giWidth*0.025);
					glVertex2f(1+fRadcentx + (2*giWidth*(0.06+(0.005*(-i-2)))), fRadcenty + i*giWidth*0.025);
				}
			glEnd();
			/////////// velocity & shield
			if ( config.getHUDType() == HUD_NETWARS )
			{
				// velocity and shield "frames"
				glBegin(GL_LINES);
					glColor3d(GLDARK_RED);
					glVertex2f(giWidth*0.02, giHeight*0.02); glVertex2f(giWidth*0.02, giHeight*0.4);
					glVertex2f(giWidth*0.02, giHeight*0.21); glVertex2f(giWidth*0.074, giHeight*0.21);
					glVertex2f(giWidth*0.1, giHeight*0.02); glVertex2f(giWidth*0.1, giHeight*0.4);
					glVertex2f(giWidth*0.1, giHeight*0.02); glVertex2f(giWidth*0.154, giHeight*0.02);
				glEnd();
				////////// velocity and shield rectangles //////////////
				glColor3d(GLGREEN);
				if ( vel != 0 )
				{
					glBegin(GL_LINE_STRIP);
						glVertex2f(giWidth*0.0335, velBegin);
						glVertex2f(giWidth*0.0605, velBegin);
						glVertex2f(giWidth*0.0605, giHeight*0.21 + velHeight);
						glVertex2f(giWidth*0.0335, giHeight*0.21 + velHeight);
						glVertex2f(giWidth*0.0335, velBegin);
					glEnd();
				}
				glBegin(GL_LINE_STRIP);
					glVertex2f(giWidth*0.1125, giHeight*0.021);
					glVertex2f(giWidth*0.1395, giHeight*0.021);
					glVertex2f(giWidth*0.1395, giHeight*0.4*PLAYER->getShield()*0.005);
					glVertex2f(giWidth*0.1125, giHeight*0.4*PLAYER->getShield()*0.005);
					glVertex2f(giWidth*0.1125, giHeight*0.021);
				glEnd();
				//// missiles
				glColor3d(GLRED);
				sprintf(str, szMissiles, PLAYER->getMissileCount());
				Text::printAt(textxpos, giHeight-iTextHeight*3, str);
				//////// powerup text outputs
				if ( (extra=PLAYER->getFirePower()) > 0 )
				{
					glColor3d(GLRED);
					Text::printAt(giWidth-iTextWidth, giHeight-iTextHeight*5, szFirePower);
					glColor3d(GLGREEN);
					i=0;
					glBegin(GL_LINES);
						do {
							glVertex2f(giWidth-iTextWidth, giHeight -iTextHeight*5.5 -i);
							glVertex2f(giWidth-iTextWidth+(iTextWidth*extra*0.01), giHeight-iTextHeight*5.5 -i);
							i++;
						} while ( (extra=extra-100) > 0 );
					glEnd();
				}
				if ( (extra=PLAYER->getCloak()) > 0 )
				{
					glColor3d(GLRED);
					Text::printAt(giWidth-iTextWidth, giHeight-iTextHeight*7, szCloak);
					glColor3d(GLGREEN);
					i=0;
					glBegin(GL_LINES);
						do {
							glVertex2f(giWidth-iTextWidth, giHeight-iTextHeight*7.5 -i);
							glVertex2f(giWidth-iTextWidth+(iTextWidth*extra*0.01), giHeight-iTextHeight*7.5 -i);
						} while ( (extra=extra-100) > 0 );
					glEnd();
				}
				if ( (extra=PLAYER->getThrust()) > 0 )
				{
					glColor3d(GLRED);
					Text::printAt(giWidth-iTextWidth, giHeight-iTextHeight*9, szThrust);
					glColor3d(GLGREEN);
					i=0;
					glBegin(GL_LINES);
						do {
							glVertex2f(giWidth-iTextWidth, giHeight-iTextHeight*9.5 -i);
							glVertex2f(giWidth-iTextWidth+(iTextWidth*extra*0.01), giHeight-iTextHeight*9.5 -i);
						} while ( (extra=extra-100) > 0 );
					glEnd();
				}
			}// HUD_NETWARS
			else if ( config.getHUDType() == HUD_ADV_NETWARS )
			{
				// shield indicator, colorful (lower) part
				float f=0, w=1;
				glBegin(GL_LINES);
				for ( f=0, w=1; f<=ansheight*((float)PLAYER->getShield()/MAX_SHIELD_CAPACITY); f++, w+=0.5*answidth/ansheight )
				{
					glColor3d(1.0, (f-anscy)/ansheight, 0.0);
					glVertex2f(anscx - w, f+anscy);
					glVertex2f(anscx + w, f+anscy);
				}
				// shield indicator, grey (upper) part
				glColor3d(0.2, 0.2, 0.2);
				for ( ; f<=ansheight; f++, w+=0.5*answidth/ansheight )
				{
					glVertex2f(anscx - w, f+anscy);
					glVertex2f(anscx + w, f+anscy);
				}
				glEnd();
				glColor3d(GLRED);
				Text::printCenteredAt(anscx, giHeight*0.22, ADNW_SHIELDS);
				Text::printCenteredAt(ansthrx, giHeight*0.22, ADNW_THRUSTERS);
				if ( PLAYER->getVelocityPercent() == 0 )
					Text::printCenteredAt(ansthrx, giHeight*0.18, ADNW_OFF);
				else
				{
					if ( PLAYER->getVelocityPercent()>0 )
						sprintf(str, "%s   %d%%", ADNW_FORWARD, PLAYER->getVelocityPercent());
					else
						sprintf(str, "%s   %d%%", ADNW_REVERSE, -PLAYER->getVelocityPercent());
					Text::printCenteredAt(ansthrx, giHeight*0.18, str);
				}
				Text::printAt(giWidth-200, iTextHeight*7, ADNW_SHELLS);
				Text::printAt(giWidth-50, iTextHeight*7, "200");
				Text::printAt(giWidth-200, iTextHeight*5, ADNW_MISSILES);
				sprintf(str, "%d", PLAYER->getMissileCount());
				Text::printAt(giWidth-50, iTextHeight*5, str);
				if ( PLAYER->getCloak() > 0 )
				{
					sprintf(str, "%d", (int)PLAYER->getCloak());
					Text::printAt(giWidth-200, iTextHeight*3, ADNW_CLOAK);
					Text::printAt(giWidth-50, iTextHeight*3, str);
				}
				if ( PLAYER->getThrust() > 0 )
				{
					sprintf(str, "%d", (int)PLAYER->getThrust());
					Text::printAt(giWidth-200, iTextHeight, ADNW_TURBO);
					Text::printAt(giWidth-50, iTextHeight, str);
				}
			}//HUD_ADV_NETWARS
			if ( PLAYER->isHit() ) // draw red screen to indicate hit
			{
				glColor4d(1.0, 0.0, 0.0, 0.4);
				glBegin(GL_QUADS);
					glVertex2f(0, 0);
					glVertex2f(giWidth, 0);
					glVertex2f(giWidth, giHeight);
					glVertex2f(0, giHeight);
				glEnd();
			}
			if ( PLAYER->isCloaked() )//////// cloak
			{
				glColor4d(0.0, 0.0, 1.0, 0.2);
				glBegin(GL_QUADS);
					glVertex2f(0, 0);
					glVertex2f(giWidth, 0);
					glVertex2f(giWidth, giHeight);
					glVertex2f(0, giHeight);
				glEnd();
			}
			///////////// missile lock arrows //////////////
			for ( i=1; pships[i]!=0 && i<MAX_SHIPS; i++ )
			{
				base = pships[i]->getPosition() - pos0;
				if ( DEGREES(acos(DotProduct(base, PLAYER->getLookDir()) / (base.getLength()) )) < 3 )
				{
					glColor3d(GLGREEN);
					glCallList(DL_LOCKARROWS);
					locked = true;	// just a flag
					PLAYER->setLockTarget(pships[i]);
					break;
				}
			}
			if ( !locked )
			{	////////// crosshair //////////////
				glColor3d(GLRED);
				glBegin(GL_LINES);
					glVertex2f(giWidthHalf, giHeightHalf - giHeight/20); glVertex2f(giWidthHalf, giHeightHalf - giHeight/8);
					glVertex2f(giWidthHalf, giHeightHalf + giHeight/20); glVertex2f(giWidthHalf, giHeightHalf + giHeight/8);
					glVertex2f(giWidthHalf - giHeight/20, giHeightHalf); glVertex2f(giWidthHalf - giHeight/8, giHeightHalf);
					glVertex2f(giWidthHalf + giHeight/20, giHeightHalf); glVertex2f(giWidthHalf + giHeight/8, giHeightHalf);
				glEnd();
				PLAYER->setLockTarget(NULL);
			}
			/*********** text outputs *************/
			///////////// fps //////////////
			if ( bShowFPS )
			{
				//tickcount = getTickCount();
				if ( tickcount-startcount > 1000 )
				{
					fFPS = framecount/((tickcount-startcount)/1000.0f);
					startcount = tickcount;
					framecount = 0;
				}
				sprintf(str, "%.2f fps", fFPS);
				glColor3d(GLWHITE);
				Text::printAt(0.0, giHeight-iTextHeight, str);
			}
			//// message
			glColor3d(GLGREEN);
			Text::printCenteredAt(giWidthHalf, MESSAGE_YCOORD, message);
			//// score
			glColor3d(GLRED);
			sprintf(str, szScore, score);
			Text::printAt(textxpos, giHeight-iTextHeight, str);
		glPopMatrix();
		glMatrixMode(GL_MODELVIEW);
		glEnable(GL_LIGHTING);
	}// !gameover_int
}

void Game::displayFunc()
{
	Vector3D					pos, rot, up, dir;
	GLfloat						lightpos[4] = {100, 0, -150, 1.0f};
	unsigned int				i;

	// color buffer clearing not necessary when using a sky box
	glClearColor(0, 0, 0, 0);
	glClearDepth(1.0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();

	glPushMatrix();
		PLAYER->cameraRotate();
		glCallList(DL_STARFIELD);
		PLAYER->cameraTranslate();
		PLAYER->drawBullets();
		glLightfv(GL_LIGHT0, GL_POSITION, lightpos);

		for ( i=0; ppowerups[i] && i<MAX_POWERUPS; i++ )
		{
			glPushMatrix();
				ppowerups[i]->draw();
			glPopMatrix();
		}

		for ( i=1; pships[i] && i<MAX_SHIPS; i++ )
		{
			glPushMatrix();
				pships[i]->draw();
			glPopMatrix();
			pships[i]->drawBullets();
		}

		for ( i = 0; i < partsys.size(); ++i )
			partsys[i].draw();

		for ( i = 0; i < missiles.size(); ++i )
		{
			glPushMatrix();
				missiles[i].draw();
			glPopMatrix();
		}
		
		for ( HitIndicatorList::iterator it = hitindicators.begin();
				it != hitindicators.end();
				++it )
		{
			glPushMatrix();
				it->draw(PLAYER->getPosition(), *PLAYER);
			glPopMatrix();
		}
	glPopMatrix();

	drawHUD();
	glFlush();
	framecount++;
	glutSwapBuffers();
	if ( glGetError() != GL_NO_ERROR )
		error("Warning: OpenGL error\n");
}

/** @brief Sets a hijacked pod free again.
  * 
  * Call this if a pirate was destroyed while stealing a pod. The pod's state
  * is set accordingly so it can return to its home position.
  *
  * @param pPirateShip The pirate that was destroyed.
  */
void freePod(SpaceObject* pPirateShip)
{
	SpaceObject* pTargetPod = ((PirateAI*)(pPirateShip->getAI()))->getTargetPod();
	if ( pTargetPod != NULL )
	{// set it free
		((PodAI*)(pTargetPod->getAI()))->setFree();
	}
}

/** @brief Adds a power-up spawning event.
  *
  * Use this method to queue a new power-up spawning event. This will use the
  * corresponding method in class Level to add the event.
  *
  * @param timeoffset The time offset from 'now'.
  * @todo More flexible events; random power-up position?
  */
void Game::addPowerupEvent(int timeoffset)
{
	LevelEvent event;
	event.time_offset = timeoffset;
	event.type = SPAWN_POWERUP;
	event.putype = PowerUpOrder[lastpowerupspawned++];
	event.postype = PLAYER_TOP;
	event.distance = PIRATE_SPAWN_DISTANCE;
	event.processed = false;
	plevel->addEvent(event);
	if ( PowerUpOrder[lastpowerupspawned] >= POWERUP_TYPE_COUNT )
		lastpowerupspawned = 0;
}

/** @brief Destroys a ship.
  *
  * Call this whenever a ship should be destroyed. If it is the player, the
  * game is over and the Game will react accordingly.
  * @param victim Array index of the ship to be destroyed.
  * @param killer Array index of the killer. Needed for score counting.
  */
void Game::destroyShip(int victim, int killer)
{// destroy ship
	if ( pships[victim]==PLAYER )
	{
		gameover_int = true;
		gameovertc = getElapsedTime();
		setMessage(GAMEOVER_TEXT);
	}
	else
	{
		partsys.push_back(ParticleSystem(pships[victim]->getPosition(), 100));
		if ( pships[victim]->getModel() == DL_PIRATE )
		{
			freePod(pships[victim]);
			if (pships[killer]==PLAYER)
				score += SCORE_ALIEN;
		}
		delete pships[victim];
		aishipcount--;
		plevel->incAIkilledCount();
		for ( int k=victim; k<MAX_SHIPS-1; k++ )
			pships[k] = pships[k+1];
		pships[MAX_SHIPS-1] = NULL;
	}
	//pSound->play_explosion();
}

/** @brief Destroys a pod.
  *
  * The pod is destroyed and the object is deleted from the game space.
  * A new ParticleSystem is spawned at its position.
  * @param victim Index to ppowerups. This is the pod to be destroyed.
  * @param notifyOthers If true, other AIs (namely the pirates) get notified
  *			about the pod destruction.
  */
void Game::destroyPod(int victim, bool notifyOthers/*=true*/)
{
	if ( notifyOthers )
	{// notify pirate AIs about pod destruction
		for ( int i=0; pships[i] && i < MAX_SHIPS; i++ )
			if ( pships[i]->getModel() == DL_PIRATE )
				((PirateAI*)(pships[i]->getAI()))->notifyPodDestruction(ppowerups[victim]);
	}
	partsys.push_back(ParticleSystem(ppowerups[victim]->getPosition(), 100));
	delete ppowerups[victim];
	for ( ; victim<MAX_POWERUPS; victim++ )
		ppowerups[victim] = ppowerups[victim+1];
	ppowerups[MAX_POWERUPS-1] = NULL;
	setMessage(szPodDead);
	podcount--;
	//pSound->play_explosion();
}

/** @brief Check objects for collisions.
  *
  * Bullet-ship collisions will damage a ship's shields and finally destroy
  * the ship.
  * Bullet-missile collisions will destroy the missile.
  * Missile-ship collisions: see bullet-ship.
  * Ship-ship collisions will leave the ships involved in the 'crash' tumbling
  * in space for a while.
  *
  * @todo Cut this method into smaller pieces.
  * @todo Increase efficiency. Here is a major slow-down if there are lots of
  * objects and bullets to be checked!
  */
void Game::detectCollisions()
{
	if ( gameover_int )
		return;

	unsigned int	i, b, c, k, power;
	float			radius;
	Vector3D		dist, pos, bulletpos, pos0 = PLAYER->getPosition();
	bool			pickup;

	for ( i=0; pships[i] && i<MAX_SHIPS; i++ )
	{// check any ship
		for ( c=0; pships[c] && c<MAX_SHIPS; c++ )
		{// check every bullet by this ship
			if ( i==c )
				continue;// don't hurt yourself...

			for ( b=0; b<MAX_BULLETS; b++ )
			{
				if ( pships[i]->m_pbullets[b]->isActive() )
				{// every bullet by ship[i] vs. ship[c]
					bulletpos = pships[i]->m_pbullets[b]->getPosition();
					dist = pships[c]->getPosition() - bulletpos;
					radius = SHIP_RADIUS+BULLET_RADIUS;
					if ( dist.getLength() <= radius )
					{// collision detected
						power = ( pships[i]->m_pbullets[b]->getModel() == DL_BULLET1) ? BULLET_POWER : 2*BULLET_POWER;
						hitindicators.push_back(HitIndicator(bulletpos));
						if ( pships[c]->hitByBullet(power)==false )	// any shield left?
						{
							destroyShip(c, i);
							if ( c > 0 ) // because of PLAYER; TODO: keep this out?
								c--;
						}
						//else if ( pships[c]==PLAYER )
							//pSound->play_hit();

						pships[i]->m_pbullets[b]->stop();
						break; // this ship[c] cannot get hit by another bullet by ship[i]
					}
				}
			}// for
		}
		/************* bullet-powerup collisions ********************/
		for ( c=0; ppowerups[c] && c<MAX_POWERUPS; c++ )
		{// check every bullet by this ship
			for ( b=0; b<MAX_BULLETS; b++ )
			{
				if ( pships[i]->m_pbullets[b]->isActive() )
				{// every bullet by ship[i] vs. powerup[c]
					bulletpos = pships[i]->m_pbullets[b]->getPosition();
					dist = ppowerups[c]->getPosition() - bulletpos;
					radius = POWERUP_RADIUS+BULLET_RADIUS;
					if ( dist.getLength() <= radius )
					{// collision detected
						if ( ppowerups[c]->hitByBullet(BULLET_POWER)==false ) // any shield left?
						{
							if ( ppowerups[c]->getModel() == DL_POD )
								destroyPod(c);
							else
							{// destroy power-up
								partsys.push_back(ParticleSystem(ppowerups[c]->getPosition(), 50));
								// pods don't reappear; now handled above
								addPowerupEvent(4000);
								delete ppowerups[c];
								for ( k=c; k<MAX_POWERUPS-1; k++ )
									ppowerups[k] = ppowerups[k+1];
								ppowerups[MAX_POWERUPS-1] = NULL;
								c--;
								//pSound->play_explosion();
							}
							pships[i]->m_pbullets[b]->stop();
							break; // this powerup[c] cannot get hit by another bullet by ship[i]
						}//if
					}//if
				}//if
			}// next bullet
		}//next powerup
		/************* bullet-missile collisions ********************/
		for ( c = 0; c < missiles.size(); ++c )
			{// check every missile
				for ( b=0; b<MAX_BULLETS; b++ )
				{
					if ( pships[i]->m_pbullets[b]->isActive() )
					{// every bullet by ship[i] vs. missile[c]
						bulletpos = pships[i]->m_pbullets[b]->getPosition();
						dist = missiles[c].getPosition() - bulletpos;
						radius = MISSILE_RADIUS+BULLET_RADIUS;
						if ( dist.getLength() <= radius )
						{// collision detected, destroy missile
							partsys.push_back(ParticleSystem(missiles[c].getPosition(), 50));
							missiles.erase(missiles.begin() + c);
							if (pships[i]==PLAYER)
								score += SCORE_MISSILE;
							pships[i]->m_pbullets[b]->stop();
							//pSound->play_explosion();
							break; // this missile[c] cannot get hit by another bullet by ship[i]
						}//if
					}//if
				}// next bullet
			}//next missile
		}

	for ( c=0; ppowerups[c] && c<MAX_POWERUPS; c++ )
	{// check for player--powerup collision (= pick up)
		pickup=true;
		dist = ppowerups[c]->getPosition() - pos0;
		radius = SHIP_RADIUS + POWERUP_RADIUS;
		if ( dist.getLength() <= radius )
		{// pick up
			switch ( ppowerups[c]->getModel() )
			{
				case DL_HEALTH: PLAYER->collectedShield(); break;
				case DL_FIREPOWER: PLAYER->collectedFirePower(); break;
				case DL_THRUST: PLAYER->collectedThrust(); break;
				case DL_XTRAMSSL: PLAYER->collectedMissiles(); break;
				case DL_CLOAK: PLAYER->collectedCloak(); break;
				default: pickup=false; break;
			}
			if ( pickup )
			{// delete this powerup
				delete ppowerups[c];
				for ( k=c; k<MAX_POWERUPS-1; k++ )
					ppowerups[k] = ppowerups[k+1];
				ppowerups[MAX_POWERUPS-1] = NULL;
				c--;
				addPowerupEvent(4000);
				//pSound->play_pickup();
			}
		}
	}

	// check missile-ship collisions
	for ( i = 0; i < missiles.size(); ++i )
		for ( c=0; pships[c] && c<MAX_SHIPS; c++ )
		{
			dist = missiles[i].getPosition() - pships[c]->getPosition();
			radius = MISSILE_RADIUS + SHIP_RADIUS;
			if ( dist.getLength() <= radius )
			{
				if ( pships[c]->hitByBullet(MISSILE_POWER) == false )	// shield left?
				{
					destroyShip(c, i);
					if ( c > 0 ) // because of PLAYER; TODO: keep this out?
						c--;
				}
				//else if ( pships[c]==PLAYER )
					//pSound->play_hit();

				missiles.erase(missiles.begin() + i);
			}
		}
	//end for

	for ( i=1; pships[i] && i<MAX_SHIPS; i++ )
	{// check ship-ship collisions (TODO: every ship? only PLAYER right now)
		dist = pships[i]->getPosition() - pos0;
		radius = SHIP_RADIUS + SHIP_RADIUS;
		if ( dist.getLength() <= radius )
		{// collision
			Vector3D v = pos-pos0;
			v.normalize();
			pships[i]->collide(v);
			PLAYER->collide(v);
		}
	}
}

/** @brief Launches a missile.
  *
  * First tries to find a target in front of pship and then fires a missile.
  * @param pship The ship that requested the missile launch.
  * @todo Treat AIs differently?
  */
void Game::launchMissile(SpaceObject* pship)
{
	Vector3D		pos0 = pship->getPosition(),
					base;
	Missile			missile;

	for ( int i=0; pships[i]!=0 && i<MAX_SHIPS; i++ )
	{
		if (pships[i] == pship)	// don't target yourself
			continue;
		base = pships[i]->getPosition() - pos0;
		if ( DEGREES(acos(DotProduct(base, pship->getLookDir()) / (base.getLength()) )) < 3 )
		{
			missile.start(pship, pships[i], DL_MISSILE);
			missiles.push_back(missile);
			pship->clearMissileFlag();
			if (pships[i]==PLAYER)
			{
				setMessage(szMsslWarn);
				//pSound->play_warning();
			}
			break;
		}
		if ( !pship->getMissileFlag() )
			break;
	}
}

void Game::stepObjects()
{
	int now, timediff;

	do 
	{// make sure there is no 'zero step' on fast computers
		now = getElapsedTime();
		timediff = now - oldtickcount;
	} while (timediff < 1);
	oldtickcount = now;

	if ( (now-lastmessagetickcount) > MESSAGE_VISIB_TC )
		setMessage("");
	
	plevel->step(timediff);

	if ( bForwardPressed )
		PLAYER->accelerate(5, now);

	if ( bBackwardPressed )
		PLAYER->accelerate(-5, now);

	if ( lbuttonDown && !gameover_int && PLAYER->shoot() )
  {
		//pSound->play_shot();
  }

	unsigned int i = 0,
		j = 0;

	for ( i=0; ppowerups[i] && i<MAX_POWERUPS; i++ )
		if ( ppowerups[i]->move(timediff) == false )
			destroyPod(i);

	for ( i=0; pships[i] && i<MAX_SHIPS; i++ )
	{
		if ( (pships[i] == PLAYER) && gameover_int)
			continue;

		if ( pships[i]->move(timediff) == false )
		{// the ship wants to be destroyed
			if ( pships[i]->getModel() == DL_PIRATE )
			{// it is a pirate, maybe has reached pod destruction distance
				SpaceObject* pTargetPod = ((PirateAI*)(pships[i]->getAI()))->getTargetPod();
				if ( pTargetPod!=NULL )
				{// pod destruction
					j=0;
					while (ppowerups[j]!=pTargetPod)	// find the pod's array position
						j++;
					destroyPod(j, false);
					setMessage(szPodDead);
				}
			}
			// destroy ship
			plevel->incAIkilledCount();
			aishipcount--;
			delete pships[i];
			for ( j=i; j<MAX_SHIPS-1; j++ )
				pships[j] = pships[j+1];
			pships[MAX_SHIPS-1] = NULL;
			continue;
		}
		if ( pships[i]->getMissileFlag() )
			launchMissile(pships[i]);
	}

	// particle systems
	for ( i=0; i < partsys.size(); ++i )
		if (partsys[i].move(timediff)==false)
			partsys.erase(partsys.begin() + i);

	// missiles
	for ( size_t m=0; m < missiles.size(); ++m )
		if ( missiles[m].move(timediff)==false )	// returns false when too old
			missiles.erase(missiles.begin() + i);

	// hit indicators
	for ( HitIndicatorList::iterator it = hitindicators.begin();
			it != hitindicators.end();
			++it )
		if ( it->move(timediff) == false )
		{// copy iterator because it is invalid after erasing
			HitIndicatorList::iterator del = it;
			--it;
			hitindicators.erase(del);
		}

	if ( gameover_ext == false )
		if ( gameover_int && (now - gameovertc) > SHOW_GAMEOVER )
			gameover_ext = true;

	if ( levelover_ext == false )
		if ( levelover_int && (now - levelovertc) > LEVEL_OVER_PLAY )
			levelover_ext = true;

	detectCollisions();
}

/** @brief Mark the level as over internally.
  *
  * This allows the player to fly around some time after the last alien is
  * destroyed.
  * @param over If true, the level is marked as over. If false, the level goes
  *             on. This should be used when switching levels.
  */
void Game::setLevelOver(bool over)
{
	levelover_int = over; 
	levelovertc = getElapsedTime();
}

/** @brief Sets the mouse position for internal calculations.
  *
  * This especially affects the player's directional control.
  * @todo Check whether the controls behave the same with different screen
  *       resolutions.
  */
void Game::setMousePos(int x, int y)
{
	mousex = x;
	mousey = giHeight-y;
	if ( fabs(giWidth/2 - x) < MOUSE_DEADZONE )
		x = giWidth/2;
	if ( fabs(giHeight/2 - y) < MOUSE_DEADZONE )
		y = giHeight/2;
	PLAYER->setAutoRotation(-((GLfloat)y / (GLfloat)giHeight-0.5)*MOUSE_SENSITIVITY,
							-((GLfloat)x / (GLfloat)giWidth-0.5)*MOUSE_SENSITIVITY, 0);
}

void Game::keyboardFunc(unsigned char key, int x, int y)
{
	switch(key)
	{
		case 'f': toggleShowFPS(); break;
		case 'w': bForwardPressed = true; break;	// forward
		case 's': bBackwardPressed = true; break;	// backward
		case 'a': PLAYER->cloakToggle(); break;
		case ' ': if (PLAYER->getLockTarget()!=NULL) PLAYER->launchMissile(); break;
		//case 'M': glutFullScreen(); break;
		case '+': fRadarZoom *= 2.0;break;
		case '-': fRadarZoom *= 0.5;break;
		case 27: shutdown(); exit(EXIT_SUCCESS); break;
	}
}

void Game::keyboardUpFunc(unsigned char key, int x, int y)
{
	switch(key)
	{
		case 'w': bForwardPressed = false; break;	// forward
		case 's': bBackwardPressed = false; break;	// backward
	}
}

void Game::specialFunc(int key, int x, int y)
{
}

void Game::specialUpFunc(int key, int x, int y)
{
}

void Game::mouseFunc(int button, int state, int x, int y)
{
	if (button==GLUT_LEFT_BUTTON)
	{
		if (state==GLUT_DOWN)
			lbuttonDown = true;
		else
			lbuttonDown = false;
	}
	if (button==GLUT_RIGHT_BUTTON)
	{
		if (state==GLUT_DOWN)
			rbuttonDown = true;
		else
			rbuttonDown = false;
	}
}

void Game::mouseMotionFunc(int x, int y)
{
	setMousePos(x, y);
}

void Game::idleFunc()
{
	stepObjects();
	glutPostRedisplay();
}
