/** @file game.h
  * @brief Game declaration.
  */

#ifndef game_include
#define game_include

#include <vector>
#include <list>
#include "iglut.h"
#include "level.h"
#include "missile.h"
#include "particlesystem.h"
#include "hitindicator.h"

class SpaceObject;

#define PLAYER pships[0]

/** @brief Represents a game space including all objects, events and
  *			interaction between them.
  *	
  * This is a very big, fat and ugly class at this time. However, it is
  * implemented in a way that for a whole single player game (a specific
  * order of levels being played) only one object of this class should be
  * constructed and used for this game. This means that the overhead between
  * levels is kept to a minimum (more or less).
  *
  * @todo A hit indicator like in NetWars (glutSolidTorus?).
  */
class Game : public IGlut
{
public:
					Game();
	virtual			~Game();
	void			startNewGame(Level* pstartlvl);
	void			startNewLevel(Level* plvl);
	void			start();
	void			stop();
	void			shutdown();
	void			addPowerup(PowerUpType type, GLdouble x, GLdouble y, GLdouble z);
	void			addPowerupEvent(int timeoffset);
	void			addAIShip(AIShipType type, GLdouble x, GLdouble y, GLdouble z);
	SpaceObject*	getFreePod();
	void			destroyShip(int victim, int killer);
	void			destroyPod(int victim, bool notifyOthers=true);
	void			setupTextSystem();
	void			setMousePos(int x, int y);
	void			setMessage(const char* msg);
	void			setLevelOver(bool over);

	/** Adds a bonus to the current score.
	  * @param bonus The bonus to be added to the score of this game.
	  */
	inline void		addBonusScore(int bonus) { score += bonus; }
	inline int		getScore() const { return score; }

	/// Toggles the frames per second display on or off.
	inline void		toggleShowFPS() { bShowFPS = !bShowFPS; }

	/// @returns A pointer to the SpaceObject representing the player.
	inline SpaceObject*		getPlayer() const { return PLAYER; }
	inline int		getAICount() const { return aishipcount; }
	inline int		getPodCount() const { return podcount; }

	/// @returns true if the the level is finished, that is, if all enemies
	/// are destroyed.
	inline bool		isLevelOver() const { return levelover_ext; }

	/// @returns true if the player got killed.
	inline bool		isGameOver() const { return gameover_ext; }

	// IGlut implementation
	virtual void	reshapeFunc();
	virtual void	displayFunc();
	virtual void	mouseFunc(int button, int state, int x, int y);
	virtual void	mouseMotionFunc(int x, int y);
	virtual void	keyboardFunc(unsigned char key, int x, int y);
	virtual void	keyboardUpFunc(unsigned char key, int x, int y);
	virtual void	specialFunc(int key, int x, int y);
	virtual void	specialUpFunc(int key, int x, int y);
	virtual void	idleFunc();

private:
	typedef std::vector<Missile> MissileVector;
	typedef std::vector<ParticleSystem> PartSysVector;
	typedef std::list<HitIndicator> HitIndicatorList;

	void			drawHUD();
	void			stepObjects();
	void			detectCollisions();
	void			initialize();
	void			launchMissile(SpaceObject*);
	void			drawRadarLine(const Vector3D& v);

	// private variables
	float			fRadcentx, fRadcenty;	// radar center point on screen
	float			fFPS;
	float			fRadarZoom;
	int				score,
					iTextWidth, iTextHeight,
					mousex, mousey, oldmx, oldmy;
	int				oldtickcount;
	int				aishipcount;
	int				podcount;
	int				lastpowerupspawned;		///< index to PowerUpOrder[]
	int				lastmessagetickcount;
	int				gameovertc;				///< when player was destroyed
	int				levelovertc;			///< when level has been marked as over internally
	long			startcount, framecount;
	SpaceObject*	ppowerups[MAX_POWERUPS];
	SpaceObject*	pships[MAX_SHIPS];	///< where PLAYER=pships[0] is always the player (camera)
	SpaceObject*	plocked;
	MissileVector	missiles;
	PartSysVector	partsys;
	HitIndicatorList hitindicators;
	Level*			plevel;
	bool			bShowFPS, bForwardPressed, bBackwardPressed, lbuttonDown, rbuttonDown;
	bool			gameover_int;	///< internal, to draw HUD accordingly
	bool			gameover_ext;	///< external, can be asked for by menu or so
	bool			levelover_int;	///< internal, to keep the game going for some seconds
	bool			levelover_ext;	///< external, can be asked for by menu or so
	char*			message;
	float			fMessageX;	///< game message x coordinate
	float			anscx; ///< Advanced NetWars HUD shield center x coordinate
	float			anscy; ///< Advanced NetWars HUD shield lower end y coordinate
	float			answidth; ///< Advanced NetWars HUD shield indicator width at top
	float			ansheight; ///< Advanced NetWars HUD shield indicator height
	float			ansthrx; ///< Advanced NetWars HUD thrusters text center x coord
};

#endif

