/** @file highscore.cpp
  * @brief Highscore implementation.
  */

#include <iostream>
#include <stdio.h>
#include <sys/stat.h>
#include "highscore.h"

Highscore::Highscore()
{
	entries.resize(HS_ENTRY_COUNT);
	phsmenu = NULL;
}

Highscore::~Highscore()
{
	delete phsmenu;
}

void Highscore::insert(int score, std::string name)
{
	HSEntry entry = {score, name};
	for ( entry_vector::iterator it=entries.begin(); it != entries.end(); it++ )
		if ( (*it).score <= score )
		{
			entries.insert(it, entry);
			break;
		}
	entries.pop_back();	// keep entry count constant
	if ( phsmenu != NULL )
	{
		delete phsmenu;
		phsmenu = NULL;
	}
	createHSMenu();
}

/** @brief Loads the high score table from disk.
  *
  * If no high score file exists, all entries are set to zero points, no name.
  * Else, as much records as possible (but not more than the size of the table)
  * are read from the file.
  */
void Highscore::load()
{
	char filename[256];
	HSEntry entry;
	unsigned int i=0;

	entry.score = 0;
	entry.name = "---";//.resize(HS_NAME_LENGTH);

	getFileName(filename);
	FILE* pFile = fopen(filename, "rb");
	if ( pFile == NULL )
	{
		for (i=0; i < entries.size(); ++i)
			entries[i] = entry;
		return;
	}

	char buf[HS_NAME_LENGTH];
	while ( !feof(pFile) && i<HS_ENTRY_COUNT )
	{
		fscanf(pFile, "%d %s\n", &(entry.score), buf);
		entry.name = buf;
		entries[i++] = entry;
		if ( feof(pFile) )
			break;
	}
	fclose(pFile);
}

/** @brief Writes the high score table to disk.
  *
  * @todo Directory creation code copy-pasted from Configuration.
  */
void Highscore::save()
{
	char filename[256];

#ifndef _MSC_VER
	char pathname[256];
	struct stat dummy;
	sprintf(pathname, "%s/.%s", getenv("HOME"), EXECUTABLE);
	if ( stat(pathname, &dummy) == -1 )
		mkdir(pathname, 0777);
#endif
	getFileName(filename);
	FILE* pFile = fopen(filename, "w");
	if ( pFile == NULL )
	{
		std::cerr << "error while trying to create high score file" << std::endl;
		return;
	}
	for ( size_t i=0; i < entries.size(); ++i )
		fprintf(pFile, "%d %s\n", entries[i].score, entries[i].name.c_str());
	fclose(pFile);
}

/** @brief Creates the high score file name.
  *
  * For Win32, the high score file will always be put into the current working
  * directory. All other systems (assumed to be *nix-like) will have the file
  * in $HOME/.netwarsgl/scores.
  */
void Highscore::getFileName(char* filename)
{
#ifdef _MSC_VER
	sprintf(filename, "%s", HS_FILENAME); // Win32: always use current path
#else
	sprintf(filename, "%s/.%s/%s", getenv("HOME"), EXECUTABLE, HS_FILENAME);
#endif
}

/** @brief Returns the high score menu (IGlut implementation).
  *
  * A call to this method will result in the menu being created if it did not
  * yet exist at that moment.
  * @returns A pointer to the Menu that displays the high score table.
  */
Menu* Highscore::getHSMenu()
{
	createHSMenu();
	return phsmenu;
}

/** @brief Creates the actual visible menu from the scores table.
  *
  * If the menu was created already, a call to this method has no effect.
  * To re-create the menu, delete the old one first and set phsmenu to NULL
  * before calling this method.
  */
void Highscore::createHSMenu()
{
	if ( phsmenu != NULL )
		return;

	phsmenu = new Menu();
	phsmenu->add(HS_TITLE, 0.5, 0.9, 0, 0, false);

	char buf[256];
	for ( size_t i=0; i < entries.size(); ++i )
	{
		sprintf(buf, "%06d .......... %20s", entries[i].score, entries[i].name.c_str());
		phsmenu->add(buf, 0.5, 0.8, 0, -i*20, false);
	}
	phsmenu->add(HS_CONTINUE, 0.5, 0.2, 0, 0, true);
}

