#include <vector>
#include "stdinclu.h"
#include "level.h"
#include "constants.h"
#include "vector3d.h"
#include "game.h"
#include "spaceobject.h"

Level::Level()
{
	timeoffset = 0;
	aieventcount = 0;
	aikilledcount = 0;
}

Level::Level(const Level& reflevel)
{
	timeoffset = 0;
	aieventcount = 0;
	aikilledcount = 0;
}

Level::~Level()
{

}

void Level::start(Game* pg)
{
	pGame = pg;
	timeoffset = 0;
}

/** @brief Schedules an event.
  *
  * Schedules the specified event for the given time offset (milliseconds).
  */
void Level::addEvent(const LevelEvent& event)
{
	LevelEvent *pe = new LevelEvent(event);
	pe->time_offset += timeoffset;
	events.push_back(pe);

	if ( pe->type == SPAWN_AISHIP )
		aieventcount++;
}

void Level::step(int timediff)
{
	timeoffset += timediff;
	// check for 'level over'
	if ( aikilledcount == aieventcount )
	{// all spawned AIs have been destroyed
		pGame->setLevelOver(true); // notify game
		aikilledcount = 0; // prevent the above condition from being met again
	}
	LevelEvent *pe;
	for (eventit it=events.begin(); it != events.end(); it++)
	{
		pe = (*it);
		if ( pe->time_offset <= timeoffset && !pe->processed )
		{
			if (pe->type==SPAWN_AISHIP && (pGame->getAICount() >= maxAICount))
				continue;		// keep number of AIs within range
			Vector3D	pos;
			switch (pe->postype)
			{
				case ABSOLUTE_POS: pos = pe->pos; break;
				case PLAYER_LEFT: pos = pGame->getPlayer()->getPosition() 
								  		- (pGame->getPlayer()->getRight()*pe->distance); break;
				case PLAYER_RIGHT: pos = pGame->getPlayer()->getPosition()
								   		+ (pGame->getPlayer()->getRight()*pe->distance); break;
				case PLAYER_FRONT: pos = pGame->getPlayer()->getPosition() 
								   		+ (pGame->getPlayer()->getLookDir()*pe->distance); break;
				case PLAYER_BACK: pos = pGame->getPlayer()->getPosition() 
								   		- (pGame->getPlayer()->getLookDir()*pe->distance); break;
				case PLAYER_TOP: pos = pGame->getPlayer()->getPosition()
								 		+ (pGame->getPlayer()->getUp()*pe->distance); break;
				case PLAYER_BOTTOM: pos = pGame->getPlayer()->getPosition()
								 		- (pGame->getPlayer()->getUp()*pe->distance); break;
				case EVENTPOS_TYPE_COUNT: /* here to avoid default */ break;
			}
			switch (pe->type)
			{
				case SPAWN_POWERUP: pGame->addPowerup(pe->putype, pos.x, pos.y, pos.z); break;
				case SPAWN_AISHIP: pGame->addAIShip(pe->aitype, pos.x, pos.y, pos.z); break;
			}
			pe->processed = true;
		}
	}
}

