/** @file main.cpp
  * @brief Some (too many) global functions, and the main() function.
  */

#include <GL/glut.h>
#include <iostream>
//#include <ClanLib/core.h>
//#include <ClanLib/application.h>
#include "iglut.h"
#include "stdinclu.h"
#include "constants.h"
#include "vector3d.h"
#include "spaceobject.h"
#include "particle.h"
#include "particlesystem.h"
#include "objects.h"
#include "level.h"
#include "levelmanager.h"
#include "game.h"
#include "menu.h"
#include "entername.h"
#include "highscore.h"
#include "configuration.h"
//#include "nwsound.h"

// TODO: why does this not work in constants.h?
const char* OPTIONS_STARSTRINGS[] = {"1000", "5000", "10000"};
const int	OPTIONS_STARSTRING_COUNT = 3;
const char* OPTIONS_HUDSTRINGS[] = {OPTIONS_HUD_NW, OPTIONS_HUD_ADNW};
const int	OPTIONS_HUDSTRING_COUNT = 2;
const char* OPTIONS_FSSTRINGS[] = {OPTIONS_FULLSCREEN_OFF, OPTIONS_FULLSCREEN_ON};
const int	OPTIONS_FSSTRING_COUNT = 2;
const char*	COMPLETE_FORMAT = "Level %d complete!";
const char*	BONUS_FORMAT = "Bonus  %d x %d";
const char* SCORE_FORMAT = "Score: %d";

/// Avoid GLUT confusion with reshaping callbacks.
bool				lastReshapeWasFullscreen = false;

// bad style: some globals that everyone else may use
GLint	giWidth;		///< Screen width. Global.
GLint	giHeight;		///< Screen height. Global.
GLint	giWidthHalf;	///< Half of giWidth. Global.
GLint	giHeightHalf;	///< Half of giHeight. Global.
/// Global Configuration object holds the settings.
Configuration	config;
/// Global NWSound object pointer.
//NWSound*		pSound;

IGlut*			glut;		///< Current game 'state'.
Menu			mainmenu;
Menu			options;
Menu			spinfo;		///< Single player start/info screen.
Menu			mpmenu;		///< Multi player menu
Menu*			plevelend;	///< Level end screen, bonus etc.
Game*			pgame;
LevelManager	lvlmgr;
Highscore		highscore;
SpaceObject*	ppods[6];	///< Displayed on the bonus screen.

SpaceObject		infopod      (DL_POD,          0,  500, -1700);
SpaceObject		infopirate   (DL_PIRATE,    -600,   90, -1700);
SpaceObject		infofighter  (DL_FIGHTER,   -200,   90, -1700);
SpaceObject		infoguardian (DL_GUARDIAN,   200,   90, -1700);
SpaceObject		infocommander(DL_COMMANDER,  600,   90, -1700);
SpaceObject		infoshield   (DL_HEALTH,    -600, -380, -1700);
SpaceObject		infomissiles (DL_XTRAMSSL,  -200, -380, -1700);
SpaceObject		infothrust   (DL_THRUST,     200, -380, -1700);
SpaceObject		infofirepower(DL_FIREPOWER,  600, -380, -1700);

/** Rotation speed of the logo in the middle of the main screen.
  * TODO: move to constants.h
  */
#define	INFO_AUTOROT	-0.2, 0.2, 0

void error(const char* msg)
{
	std::cerr << msg;
}

/** @brief OpenGL initialisation code.
  *
  * Here the global OpenGL states are set. This includes matrices,
  * lighting, face culling, material and blending setup.
  */
void initOpenGL()
{
	GLfloat		lightpos[4] =		{100, 0, 150, 1.0f};
	GLfloat		ambientLight[4] =	{0.3f, 0.3f, 0.3f, 1.0f};
	GLfloat		diffuseLight[4] =	{0.7f, 0.7f, 0.7f, 1.0f};
	GLfloat		specular[4] =		{1, 1, 1, 1};
	GLfloat		specreflect[4] =	{1, 1, 1, 1};
	//GLfloat		diffuseMaterial[4] = { 0.5, 0.5, 0.5, 1.0 };
	//GLfloat		mat_specular[] = { 1.0, 1.0, 1.0, 1.0 };
	//GLfloat		light_position[] = { 6.0, 1.0, 10.0, 0.0 };

	glMatrixMode(GL_MODELVIEW); 
	glEnable(GL_LIGHTING);		// lighting
	glLightfv(GL_LIGHT0, GL_AMBIENT, ambientLight);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuseLight);
	glLightfv(GL_LIGHT0, GL_SPECULAR, specular);
	glLightfv(GL_LIGHT0, GL_POSITION, lightpos);
	glEnable(GL_LIGHT0);
	glEnable(GL_DEPTH_TEST);	// depth testing
	glEnable(GL_COLOR_MATERIAL);
	glFrontFace(GL_CCW);
	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specreflect);
	glMateriali(GL_FRONT_AND_BACK, GL_SHININESS, 128);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

/** @brief Initialises the main menu with its entries and objects.
  *
  * @param pMainmenu Points to the Menu object that represents the main menu.
  * @param pTitle The SpaceObject to be displayed on top of the screen.
  * @param pLogo The SpaceObject to be displayed in the center of the screen.
  */
void initMainMenu(Menu* pMainmenu, SpaceObject* pTitle, SpaceObject* pLogo)
{
	pMainmenu->add(MAIN_SPMODE_TEXT, 0.5, 0.0, 0, MAINMENU_Y0, true);
	pMainmenu->add(MAIN_MPMODE_TEXT, 0.5, 0.0, 0, MAINMENU_Y0 - MAINMENU_ITEM_DISTANCE, true);
	pMainmenu->add(MAIN_HIGHSCORE_TEXT, 0.5, 0.0, 0, MAINMENU_Y0 - 2*MAINMENU_ITEM_DISTANCE, true);
	pMainmenu->add(MAIN_OPTIONS_TEXT, 0.5, 0.0, 0, MAINMENU_Y0 - 3*MAINMENU_ITEM_DISTANCE, true);
	pMainmenu->add(MAIN_QUIT_TEXT, 0.5, 0.0, 0, MAINMENU_Y0 - 4*MAINMENU_ITEM_DISTANCE, true);

	pMainmenu->add(MAIN_VERSION_TEXT, 0.5, 0.0, 0, MAINMENU_Y0 - 6*MAINMENU_ITEM_DISTANCE, false);
	pMainmenu->add(MAIN_CONTACT_TEXT, 0.5, 0.0, 0, MAINMENU_Y0 - 7*MAINMENU_ITEM_DISTANCE, false);

	// add objects
	pMainmenu->add(pTitle);
	pLogo->setAutoRotation(0, 0.05, 0);
	pMainmenu->add(pLogo);
}

/** @brief Initialises the Options menu with its entries and objects.
  */
void initOptionsMenu(Menu* pOptionsMenu, SpaceObject* pTitle, SpaceObject* pStarfield)
{
	MenuItem stars(OPTIONS_STARS_TEXT, 0.5, 0.0, 0, OPTIONS_Y0, true);
	MenuItem hud(OPTIONS_HUD_TEXT, 0.5, 0.0, 0, OPTIONS_Y0 - OPTIONS_ITEM_DISTANCE, true);
	MenuItem fullscreen(OPTIONS_FULLSCREEN_TEXT, 0.5, 0.0, 0, OPTIONS_Y0-2*OPTIONS_ITEM_DISTANCE, true);
	MenuItem back(OPTIONS_BACK_TEXT, 0.5, 0.0, 0, OPTIONS_Y0 - 3*OPTIONS_ITEM_DISTANCE, true);

	stars.setToggleValues(OPTIONS_STARSTRINGS, OPTIONS_STARSTRING_COUNT);
	stars.setToggleSelection(config.getNumStars());
	hud.setToggleValues(OPTIONS_HUDSTRINGS, OPTIONS_HUDSTRING_COUNT);
	hud.setToggleSelection(OPTIONS_HUDSTRINGS[config.getHUDType()]);
	fullscreen.setToggleValues(OPTIONS_FSSTRINGS, OPTIONS_FSSTRING_COUNT);
	fullscreen.setToggleSelection(OPTIONS_FSSTRINGS[config.isFullScreen() ? 1 : 0]);

	pOptionsMenu->add(OPTIONS_TITLE_TEXT, 0.5, 0.5, 0, 0, false);
	pOptionsMenu->add(stars);
	pOptionsMenu->add(hud);
	pOptionsMenu->add(fullscreen);
	pOptionsMenu->add(back);

	// objects
	pOptionsMenu->add(pStarfield);
	pOptionsMenu->add(pTitle);
}

/** @brief Initialises the single player info screen.
  */
void initInfoScreen()
{
	spinfo.add(INFO_PODS_TEXT,   0.5, 0.95, 0, 0, false);
	spinfo.add(INFO_ALIENS_TEXT, 0.5, 0.7, 0, 0, false);
	spinfo.add(INFO_SATS_TEXT,   0.5, 0.35, 0, 0, false);
	spinfo.add(INFO_START_TEXT,  0.5, 0.05, 0, 0, true);

	spinfo.add(INFO_PIRATE,    0.2, 0.45, 0, 0, false);
	spinfo.add(INFO_FIGHTER,   0.4, 0.45, 0, 0, false);
	spinfo.add(INFO_GUARDIAN,  0.6, 0.45, 0, 0, false);
	spinfo.add(INFO_COMMANDER, 0.8, 0.45, 0, 0, false);

	spinfo.add(INFO_SHIELD,    0.2, 0.1, 0, 0, false);
	spinfo.add(INFO_MISSILES,  0.4, 0.1, 0, 0, false);
	spinfo.add(INFO_THRUST,    0.6, 0.1, 0, 0, false);
	spinfo.add(INFO_FIREPOWER, 0.8, 0.1, 0, 0, false);

	spinfo.add(&infopod);
	spinfo.add(&infopirate);
	spinfo.add(&infofighter);
	spinfo.add(&infoguardian);
	spinfo.add(&infocommander);
	spinfo.add(&infoshield);
	spinfo.add(&infomissiles);
	spinfo.add(&infothrust);
	spinfo.add(&infofirepower);
}

void initMPMenu()
{
	MenuItem join(MP_JOIN_TEXT, 0.5, 0.5, 0, 0, true);
	MenuItem host(MP_HOST_TEXT, 0.5, 0.5, 0, -20, true);
	MenuItem back(MP_BACK_TEXT, 0.5, 0.5, 0, -40, true);
	mpmenu.add(join);
	mpmenu.add(host);
	mpmenu.add(back);
}

void createLevelEndScreen()
{
	char		completebuf[256], bonusbuf[256], scorebuf[256];

	sprintf(completebuf, COMPLETE_FORMAT, lvlmgr.getCurrentLevelNumber());
	sprintf(bonusbuf, BONUS_FORMAT, pgame->getPodCount(), lvlmgr.getCurrentLevelNumber()*POD_BONUS_MULT);
	sprintf(scorebuf, SCORE_FORMAT, pgame->getScore());

	plevelend = new Menu();

	MenuItem complete(completebuf, 0.5, 0.75, 0, 0, false);
	MenuItem bonus(bonusbuf, 0.5, 0.25, 0, 0, false);
	MenuItem score(scorebuf, 0.5, 0.25, 0, -20, false);
	MenuItem cont(LEVEL_CONTINUE_TEXT, 0.5, 0.1, 0, 0, true);
	for ( int i=0; i < pgame->getPodCount(); i++ )
		plevelend->add(ppods[i]);

	plevelend->add(complete);
	plevelend->add(bonus);
	plevelend->add(score);
	plevelend->add(cont);
}

// global IGlut implementation/callers
void reshape(int width, int height)
{
	glViewport (0, 0, (GLsizei) width, (GLsizei) height);
	glMatrixMode (GL_PROJECTION);
	glLoadIdentity();
	GLdouble aspect = (GLdouble) width / (GLdouble) height;
	gluPerspective(45.0, aspect, 1.0, 300000.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	if ( !lastReshapeWasFullscreen )
	{
		config.setWidth(width);
		config.setHeight(height);
	}
	giWidth = width;
	giHeight = height;
	giWidthHalf = width/2;
	giHeightHalf = height/2;
	glut->reshapeFunc();
	mainmenu.reshapeFunc();
	options.reshapeFunc();
	spinfo.reshapeFunc();
	mpmenu.reshapeFunc();
	createLockArrows();
	lastReshapeWasFullscreen = false;
}

void displayFunc()
{
	glut->displayFunc();
}

void mouseFunc(int button, int state, int x, int y)
{
	glut->mouseFunc(button, state, x, y);
}

void mouseMotionFunc(int x, int y)
{
	glut->mouseMotionFunc(x, y);
}

void keyboardFunc(unsigned char key, int x, int y)
{
	if ( key == 27 )
	{// handle ESC for different situtations
		if ( glut == &mainmenu )
		{
			exit(EXIT_SUCCESS);
		}

		if ( glut == pgame )
		{
			pgame->stop();
			glut = &mainmenu;
			delete pgame;
			glutSetCursor(GLUT_CURSOR_LEFT_ARROW);
		}
		else if ( glut==&options || glut==&spinfo || glut==&mpmenu )
		{
			glut = &mainmenu;
		}
	}
	glut->keyboardFunc(key, x, y);
}

void keyboardUpFunc(unsigned char key, int x, int y)
{
	glut->keyboardUpFunc(key, x, y);
}

void specialFunc(int key, int x, int y)
{
	glut->specialFunc(key, x, y);
}

void specialUpFunc(int key, int x, int y)
{
	glut->specialUpFunc(key, x, y);
}

/** @brief Global GLUT idle function.
  *
  * This function is registered as the GLUT idle callback. It forwards the
  * call to the current IGlut implementation pointed to by @c glut. It also
  * handles every special action, like switching the menus and checking
  * whether a game has ended.
  *
  * @todo Too long. Separate into multiple functions. Extra file maybe?
  */
void idleFunc()
{
	///////// main menu
	if ( glut==&mainmenu && mainmenu.isItemActivated() )
	{// TODO: menu item names in file, load to (hash) table or map
		const char *activated = mainmenu.getActivatedText();
		if ( strcmp(activated, MAIN_SPMODE_TEXT)==0 )
			glut = &spinfo;
		else if ( strcmp(activated, MAIN_MPMODE_TEXT)==0 )
			glut = &mpmenu;
		else if ( strcmp(activated, MAIN_HIGHSCORE_TEXT)==0 )
			glut = (Menu*)highscore.getHSMenu();
		else if ( strcmp(activated, MAIN_OPTIONS_TEXT)==0 )
			glut = &options;
		else if ( strcmp(activated, MAIN_QUIT_TEXT)==0 )
			exit(EXIT_SUCCESS);
		mainmenu.resetItemActivated();
	}
	///////// info screen
	else if ( glut==&spinfo && spinfo.isItemActivated() )
	{// currently only one item
		lvlmgr.reset();
		glutSetCursor(GLUT_CURSOR_NONE);
		pgame = new Game();
		pgame->startNewGame(lvlmgr.getCurrentLevel());
		pgame->reshapeFunc();
		glut = pgame;
		pgame->start();
	}
	///////// multi player menu
	else if ( glut==&mpmenu && mpmenu.isItemActivated() )
	{
		const char *activated = mpmenu.getActivatedText();
		if ( strcmp(activated, MP_BACK_TEXT) == 0 )
		{
			glut = &mainmenu;
		}
		mpmenu.resetItemActivated();
	}
	///////// options menu
	else if ( glut==&options && options.isItemActivated() )
	{
		const char* itemtext = options.getActivatedText();
		if ( strstr(itemtext, OPTIONS_STARS_TEXT) != NULL )
		{// change number of stars
			config.setNumStars( atoi(options.getToggleValue(itemtext)) );
			createStarfield();
		}
		else if ( strstr(itemtext, OPTIONS_HUD_TEXT) != NULL )
		{// change HUD type
			if ( strcmp(options.getToggleValue(itemtext), OPTIONS_HUD_NW) == 0 )
				config.setHUDType(HUD_NETWARS);
			else if ( strcmp(options.getToggleValue(itemtext), OPTIONS_HUD_ADNW) == 0 )
				config.setHUDType(HUD_ADV_NETWARS);
		}
		else if ( strstr(itemtext, OPTIONS_FULLSCREEN_TEXT) != NULL )
		{// fullscreen selected
			if ( strcmp(options.getToggleValue(itemtext), OPTIONS_FULLSCREEN_ON)==0 )
			{
				lastReshapeWasFullscreen = true;
				glutFullScreen();
				config.setFullScreen(true);
			}
			else
			{
				glutReshapeWindow(config.getWidth(), config.getHeight());
				config.setFullScreen(false);
			}
		}
		else if ( strcmp(itemtext, OPTIONS_BACK_TEXT)==0 )
		{
			options.resetItemActivated();
			glut = &mainmenu;
		}
		options.deactivateItem();
	}
	///////// level end screen
	else if ( glut==plevelend && plevelend->isItemActivated() )
	{// currently has only one item
		glutSetCursor(GLUT_CURSOR_NONE);
		lvlmgr.nextLevel();
		pgame->startNewLevel(lvlmgr.getCurrentLevel());
		glut = pgame;
		delete plevelend;
		pgame->start();
	}
	///////// game
	else if ( glut==pgame && pgame->isGameOver() )
	{
		pgame->stop();
		if ( pgame->getScore() > highscore.getMinimumScore() )
		{
			glut = (IGlut*)highscore.getEnterName();
			glutPostRedisplay();
		}
		else
		{
			glut = highscore.getHSMenu();
			delete pgame;
			glutSetCursor(GLUT_CURSOR_LEFT_ARROW);
		}
	}
	///////// game - level over
	else if ( glut==pgame && pgame->isLevelOver() )
	{
		pgame->stop();
		pgame->addBonusScore(pgame->getPodCount() * lvlmgr.getCurrentLevelNumber() * POD_BONUS_MULT);
		createLevelEndScreen();
		glut = plevelend;
		glutSetCursor(GLUT_CURSOR_LEFT_ARROW);
	}
	///////// enter name
	else if ( glut==(IGlut*)(highscore.getEnterName()) && ((EnterName*)glut)->isReady() )
	{
		highscore.insert(pgame->getScore(), ((EnterName*)glut)->getName());
		glut = highscore.getHSMenu();
		delete pgame;
		glutSetCursor(GLUT_CURSOR_LEFT_ARROW);
	}
	///////// high score table
	else if ( glut==highscore.getHSMenu() && ((Menu*)glut)->isItemActivated() )
	{
		((Menu*)glut)->deactivateItem();
		glut = &mainmenu;
	}
	else
		glut->idleFunc();
}

void visibilityFunc(int vis)
{
	if (vis == GLUT_VISIBLE)
		glutIdleFunc(idleFunc);
	else
		glutIdleFunc(NULL);
}

void atexitfunc()
{
	config.save();
	highscore.save();
	//delete pSound;
	//CL_SetupCore::deinit();
}

/** @brief Required to allow ClanLib to find the entry point.
  */
//class NetWarsGLApp: public CL_ClanApplication
//{
	/*virtual*/ int main(int argc, char** argv)
	{
		/*try
		{
			CL_SetupCore::init();
			//pSound = new NWSound();
		} 
		catch (CL_Error err)
		{
			std::cout << "error: " << err.message.c_str() << std::endl;
		}*/

		SpaceObject title(DL_TITLE, 0, 160, -500);
		SpaceObject caldera(DL_LOGO, 0, 0, -700);
		SpaceObject starfield(DL_STARFIELD, 0, 0, 0);
		starfield.setAutoRotation(0, 0.005, 0);
		for (int i=0; i < 6; i++ )
		{
			ppods[i] = new SpaceObject(DL_POD, (i*POWERUP_RADIUS*2)-6*POWERUP_RADIUS, 0, -1500);
			ppods[i]->setAutoRotation(POD_AUTOROTATION);
		}
		// info screen objects autorotation
		infopod.setAutoRotation(INFO_AUTOROT);
		infopirate.setAutoRotation(INFO_AUTOROT);
		infofighter.setAutoRotation(INFO_AUTOROT);
		infoguardian.setAutoRotation(INFO_AUTOROT);
		infocommander.setAutoRotation(INFO_AUTOROT);
		infoshield.setAutoRotation(INFO_AUTOROT);
		infomissiles.setAutoRotation(INFO_AUTOROT);
		infothrust.setAutoRotation(INFO_AUTOROT);
		infofirepower.setAutoRotation(INFO_AUTOROT);

		std::cout << "NetWarsGL, written by Ingmar Frank, (c) 2003, 2004" << std::endl;
		glutInit(&argc, argv);

		config.load();
		highscore.load();

#ifdef _MSC_VER
		initTimer();
#endif

		atexit(atexitfunc);

		glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
		giWidth = config.getWidth(); giWidthHalf = giWidth/2;
		giHeight = config.getHeight(); giHeightHalf = giHeight/2;
		glutInitWindowSize(giWidth, giHeight); 
		glutCreateWindow(argv[0]);

		//glutGameModeString("640x480:32@60") ;
		//if (glutGameModeGet(GLUT_GAME_MODE_WIDTH) != -1)
		//	glutEnterGameMode();

		initOpenGL();
		initMainMenu(&mainmenu, &title, &caldera);
		initOptionsMenu(&options, &title, &starfield);
		initMPMenu();
		createObjects();
		initInfoScreen();

		glut = &mainmenu;

		glutReshapeFunc(reshape);
		glutDisplayFunc(displayFunc);
		glutVisibilityFunc(visibilityFunc);
		glutSpecialFunc(specialFunc);
		glutKeyboardFunc(keyboardFunc);
		glutKeyboardUpFunc(keyboardUpFunc);
		glutMouseFunc(mouseFunc);
		glutMotionFunc(mouseMotionFunc);
		glutPassiveMotionFunc(mouseMotionFunc);
		if ( config.isFullScreen() )
		{
			lastReshapeWasFullscreen = true;
			glutFullScreen();
		}
		glutMainLoop();
		return 0;
	}

/*	virtual char* get_title()
	{
		return "NetWarsGL";
	}

} app; // class NetwarsglApp
*/
