#include <vector>
#include <iostream>
#include "stdinclu.h"
#include "constants.h"
#include "menuitem.h"
#include "spaceobject.h"
#include "text.h"
#include "menu.h"

extern int giWidth, giHeight;

Menu::Menu()
    : selected(-1), // use -1 to show uninitialized state
    oldtickcount(0),
    itemactivated(false),
    indexactivated(-1)
{
}

Menu::~Menu()
{
}

/** @brief Adds a menu item to the menu.
  *
  * @param text The text being used for the menu item.
  * @param xmult Window width multiplier for the item's x coordinate.
  *					Typically between 0.0 and 1.0.
  * @param ymult Window height multiplier for the item's y coordinate.
  *					Typically between 0.0 and 1.0.
  * @param xoffs Offset applied to the x coordinate after multiplication.
  * @param yoffs Offset applied to the y coordinate after multiplication.
  * @param selectable If false, this item will not be selectable. It will
  *			appear as static text on the screen.
  */
void Menu::add(const char* text, float xmult, float ymult, int xoffs, int yoffs, bool selectable)
{
    MenuItem item(text, xmult, ymult, xoffs, yoffs, selectable);
    add(item);
}

/** @brief Adds a specific (pre-constructed) menu item to the menu.
  *
  * @param item The specific item to be added to the menu.
  */
void Menu::add(MenuItem item)
{
    item.calcRect();
    items.push_back(item);
    if ( selected<0 && item.isSelectable() )    // initialized?
        selected = items.size() - 1;
}

/** @brief Adds a specific object to the menu.
  *
  * @param pobj Pointer to the object to be added.
  * @remark This menu does <i>not</i> take ownership of the object, which also
  *         means that it is not deleted by the destructor.
  *         It is up to the caller to manage the object's life cycle.
  */
void Menu::add(SpaceObject* pobj)
{
    objects.push_back(pobj);
}

/** @brief Finds a menu item present at a specific position on screen.
  *
  * The whole sequence of menu items is searched for an item at the
  * specified position. If items overlap, the first one found is returned.
  * @param posx X coordinate of the position of interest.
  * @param posy Y coordinate of the position of interest.
  * @returns The index of the first item found at the specified position.
  *          If there is no item at the position, returns -1.
  */
int Menu::itemAt(int posx, int posy)
{
    RECT rect;
    int i = 0;

    for (MenuItemVector::iterator it = items.begin(); it != items.end(); ++it, ++i)
    {
        if ( (*it).isSelectable() == false )
            continue;

        rect = (*it).getRect();
        if (posx >= rect.left &&
            posx <= rect.right &&
            posy <= rect.top &&     // OpenGL y coordinates!
            posy >= rect.bottom )
            return i;
    }
    return -1;
}

/** @brief Select an item from a specific screen position.
  *
  * If an item is found by itemAt(), it may be selected.
  * @param posx X coordinate of mouse position.
  * @param posy Y coordinate of mouse position.
  * @param select If true and an item is found, it will be selected.
  */
void Menu::mouseOver(int posx, int posy, bool select)
{
    if ( select )
    {
        int sel = itemAt(posx, posy);
        if ( sel >= 0 )
            selected = sel;
    }
}

/** @brief Select the previous item in the menu item sequence.
  *
  * Uses wrap-around, so that the last item is selected if at the moment
  * of calling this method the first one was selected.
  */
void Menu::selectPrevious()
{
    do
    {
        if ( --selected < 0 )
            selected = items.size() - 1;
    } while ( !(items[selected].isSelectable()) );
}

/** @brief Select the next item in the menu item sequence.
  *
  * Uses wrap-around, so that the first item is selected if at the moment
  * of calling this method the last one was selected.
  */
void Menu::selectNext()
{
    do
    {
        if ( (unsigned)++selected == items.size() )
            selected = 0;
    } while ( !(items[selected].isSelectable()) );
}

void Menu::reshapeFunc()
{
    for ( unsigned int i = 0; i<items.size(); ++i )
        items[i].calcRect();
}

void Menu::displayFunc()
{
    RECT r;
    GLfloat lightpos[4] = {100, 0, 150, 1.0f};

    int now, timediff;

    do
    { // make sure there is no 'zero step' on fast machines
        now = getElapsedTime();
        timediff = now - oldtickcount;
    } while (timediff < 1);
    oldtickcount = now;

    glClearColor(0, 0, 0, 0);
    glClearDepth(1.0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glLoadIdentity();
    glLightfv(GL_LIGHT0, GL_POSITION, lightpos);

    // draw 3d stuff
    glPushMatrix();
    glLoadIdentity();
    for ( SpaceObjectVector::iterator oit = objects.begin(); oit!=objects.end(); ++oit )
    {
        glPushMatrix();
        (*oit)->move(timediff);
        (*oit)->draw();
        glPopMatrix();
    }
    glPopMatrix();

    // draw 2d stuff
    glPushAttrib(GL_LIGHTING_BIT);
    glDisable(GL_LIGHTING);
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    gluOrtho2D(0, giWidth, 0, giHeight);
    glColor3f(GLWHITE);

    // print all menu item texts
    for (MenuItemVector::iterator mit = items.begin(); mit!=items.end(); ++mit)
        Text::printAt((*mit).getPosX(), (*mit).getPosY(), (*mit).getText());

    // If an item is selected (highlighted, "mouse-over") draw a frame around it.
    if ( (unsigned)selected < items.size() )
    {
        glColor3f(GLGREEN);
        glBegin(GL_LINE_STRIP);
        r = items.at(selected).getRect();
        glVertex2i(r.left, r.bottom);
        glVertex2i(r.right, r.bottom);
        glVertex2i(r.right, r.top);
        glVertex2i(r.left, r.top);
        glVertex2i(r.left, r.bottom);
        glEnd();
    }
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopAttrib();

    glutSwapBuffers();
}

void Menu::mouseFunc(int button, int state, int x, int y)
{
    int sel;
    if ( state==GLUT_DOWN && (sel = itemAt(x, giHeight - y)) >= 0 )
    {
        if ( items[sel].isToggle() )
        {
            if ( button==GLUT_LEFT_BUTTON )
                items[sel].toggleForward();
            else
                items[sel].toggleBackward();
        }
        itemactivated = true;
        indexactivated = sel;
    }
}

void Menu::mouseMotionFunc(int x, int y)
{
    mouseOver(x, giHeight - y, true);
}

void Menu::keyboardFunc(unsigned char key, int x, int y)
{
    if ( key==13 || key==32 )
    {// ENTER or SPACE BAR select an item
        if ( selected >= 0 && items[selected].isToggle() )
            items[selected].toggleForward();
        // always activate the item
        itemactivated = true;
        indexactivated = selected;
    }
}

void Menu::specialFunc(int key, int x, int y)
{
    switch ( key )
    {
    case GLUT_KEY_UP: selectPrevious(); break;
    case GLUT_KEY_DOWN: selectNext(); break;

    default: break;
    }
}

void Menu::specialUpFunc(int key, int x, int y)
{
}

void Menu::keyboardUpFunc(unsigned char key, int x, int y)
{
}

void Menu::idleFunc()
{
    glutPostRedisplay();
}

/** @brief Resets the menu item activation state.
  *
  * This means that no item is activated and the first selectable item
  * is selected. Useful when switching between multiple menus.
  */
void Menu::resetItemActivated()
{
    itemactivated = false;
    for ( unsigned int i = 0; i<items.size(); ++i )
        if ( items[i].isSelectable() )
        {
            selected = i;
            break;
        }
}

/** @brief Returns the current value of a toggle menu item.
  *
  * The value is the non-static text of a toggle item that is changed
  * by every item activation.
  * @param itemtext The static text of the item of which you want to know
  *				the value.
  * @returns A pointer to the value text.
  */
const char* Menu::getToggleValue(const char* itemtext)
{
    for ( unsigned int i = 0; i<items.size(); ++i )
        if ( strstr(items[i].getText(), itemtext) != 0 )
            return items[i].getToggleValue();
    return NULL;
}

