/** @file menu.h
  * @brief Menu declaration.
  */

#ifndef menu_include
#define menu_include

#include "iglut.h"
#include "menuitem.h"

class SpaceObject;

/** @brief A fullscreen menu consisting of menu items, 3D space objects and
  * text.
  *
  * A menu can be seen as a 'game state' itself. It covers the whole screen
  * and all user interaction happens only with the menu as long as it is
  * visible. A menu holds menu items that can be selected by using the cursor
  * keys or the mouse. They can be activated by pressing RETURN or SPACE or
  * clicking the mouse. Additionally, 3D objects can be added to a menu, and
  * non-selectable items can be used as static text.
  *
  * @todo Modification / removal of items.
  */
class Menu : public IGlut
{
public:
    Menu();
    virtual ~Menu();
    void                add(const char*, float, float, int, int, bool);
    void                add(MenuItem item);
    void                add(SpaceObject* obj);
    int                 itemAt(int posx, int posy);
    void                mouseOver(int posx, int posy, bool select);
    void                selectPrevious();
    void                selectNext();
    void                resetItemActivated();
    const char*         getToggleValue(const char* itemtext);

    /** @brief Set the currently selected item.
      * @param num Index of the menu item to be selected.
      */
    inline void         setSelected(unsigned int num) { selected = num; }

    /// @returns The index of the currently selected menu item.
    inline unsigned int getSelected() const { return selected; }

    /// @returns true if an item got activated (like, by clicking on it).
    inline bool         isItemActivated() const { return itemactivated; }

    /// @returns The text of the activated item.
    inline const char*  getActivatedText() const { return items[indexactivated].getText(); }

    /** @brief Deactivates the activated item again.
      *
      * Call this after you have taken
      * an action as a reaction to an item activation (see isItemActivated())
      * to prevent a reaction from being executed twice.
      */
    inline void         deactivateItem() { itemactivated = false; }

    // IGlut implementation
    virtual void        reshapeFunc();
    virtual void        displayFunc();
    virtual void        mouseFunc(int button, int state, int x, int y);
    virtual void        mouseMotionFunc(int x, int y);
    virtual void        keyboardFunc(unsigned char key, int x, int y);
    virtual void        keyboardUpFunc(unsigned char key, int x, int y);
    virtual void        specialFunc(int key, int x, int y);
    virtual void        specialUpFunc(int key, int x, int y);
    virtual void        idleFunc();
private:
    typedef std::vector<MenuItem>       MenuItemVector;
    typedef std::vector<SpaceObject*>   SpaceObjectVector;

    MenuItemVector      items;          ///< The sequence containing all added menu items.
    SpaceObjectVector   objects;        ///< The 3d objects sequence.
    int                 selected;       ///< The currently selected item (highlighted).
    int                 oldtickcount;   ///< The last step's tick count.
    bool                itemactivated;  ///< True if an item has been clicked.
    int                 indexactivated; ///< Index of the clicked item.
};

#endif
