#include "text.h"
#include "menuitem.h"

extern int giWidth, giHeight;

MenuItem::MenuItem(const char* text,
					float xmult,
					float ymult,
					int xoffs,
					int yoffs,
					bool selectable)
					: textbase(text),
            text(text),
						xmult(xmult),
						ymult(ymult),
						xoffs(xoffs),
						yoffs(yoffs),
						selectable(selectable)
{
	this->posx = (int)((giWidth*xmult) + xoffs);
	this->posy = (int)((giHeight*ymult) + yoffs);
	rect.left = posx;
	rect.top = rect.bottom = posy;
	toggle = false;
	togsel = 0;
}

MenuItem::~MenuItem()
{

}

/** @brief Calculate the bounding rectangle.
  *
  * The position and the text (which may vary if this is a toggle
  * menu item) are taken into account for this calculation.
  * Call this every time the item text is changed.
  */
void MenuItem::calcRect()
{
	int width = Text::getTextWidth(text.c_str());

	posx = (int)((giWidth*xmult) + xoffs);
	posx -= width/2;	// center horizontally
	posy = (int)((giHeight*ymult) + yoffs);
	rect.left = posx - 2;
	rect.top = rect.bottom = posy-4;
	
	rect.top = rect.bottom + (Text::getTextHeight()+2);
	rect.right = rect.left + width + 4;
}

/** @brief Sets the values for a toggle menu item.
  * 
  * By calling this method, the menu item becomes a toggle that cycles
  * through the values when being clicked on.
  * @param vals Pointer to an array of C strings that become the toggle
  * 			values.
  * @param num The count of items in vals.
  */
void MenuItem::setToggleValues(const char** vals, int num)
{
	values.resize(num);
	for ( int i=0; i<num; i++ )
		values[i] = vals[i];
	toggle = true;
	togsel = 0;
	text = textbase + values[togsel];
	calcRect();
}

/** @brief Toggles the selection by selecting the next value.
  *
  * If the value that was selected when calling this method was the last one
  * in the list, the first one will be selected (wrap-around).
  * This method also calls calcRect().
  */
void MenuItem::toggleForward()
{
	togsel++;
	if ( togsel >= values.size() )
		togsel = 0;
	text = textbase + values[togsel];
	calcRect();
}

/** @brief Toggles the selection by selecting the previous value.
  *
  * If the value that was selected when calling this method was the first one
  * in the list, the last one will be selected (wrap-around).
  * This method also calls calcRect().
  */
void MenuItem::toggleBackward()
{
	if ( togsel == 0 )
		togsel = values.size()-1;
	else
		togsel--;
	text = textbase + values[togsel];
	calcRect();
}

/** @brief Select a specific toggle value from the list.
  *
  * If this menu item is no toggle or there is no such value as the wanted one
  * in the list, this method will have no effect.
  * @param wanted The text that should be selected.
  */
void MenuItem::setToggleSelection(const char* wanted)
{
	if ( !toggle )
		return;

	for ( unsigned int i=0; i<values.size(); ++i )
		if ( strcmp(wanted, values[i].c_str()) == 0 )
		{
			togsel = i;
			text = textbase + values[togsel];
			calcRect();
			return;
		}
}

/** @brief Select a specific toggle value from the list.
  *
  * If this menu item is no toggle or there is no such value as the wanted one
  * in the list, this method will have no effect.
  * @param wanted The value that should be selected; it will be converted to a
  * 			string.
  */
void MenuItem::setToggleSelection(int wanted)
{
	if ( !toggle )
		return;

	char buf[16];
	sprintf(buf, "%d", wanted);
	setToggleSelection(buf);
}

