#include <stdio.h>
#include <iostream>
#include <GL/glut.h>
#include <cmath>
#include "vector3d.h"
#include "nshmodel.h"

#define DEGREES(x)			((float)(x*57.2957795130823208767981548141052f))

NSHModel::NSHModel()
{
	shape.pVertices = NULL;
	shape.pPolys = NULL;
}

NSHModel::~NSHModel()
{
	if (shape.pVertices)
		delete[] shape.pVertices;
	if (shape.pPolys)
		delete[] shape.pPolys;
}

/** @brief Parse NSH model data.
  *
  * Extracts the model data from the array pointed to by pData
  * and puts it into the structures.
  * @param pData Pointer to the contents of the NSH model file.
  * @return false if anything went wrong. In this case, no OpenGL display
  * 		list should be generated from this model data.
  *
  * @todo Check security. Improve speed. Make it a real parser.
  */
bool NSHModel::parseData(const char *pData)
{
	// begin with the SHAPE, and extract the name first
	char *pPtr = strstr(pData, "SHAPE");
	if ( pPtr == NULL )
	{
		std::cerr << "not a valid NSH model" << std::endl;
		return false;
	}
	pPtr += sizeof("SHAPE")-1;
	while (*pPtr <= 32) // interpret all characters below #32 as whitespaces
		pPtr++;			// and strip them

	char *pPtr2 = strstr(pPtr, "_shape");	// does not belong to the name
	if ( pPtr2 == NULL )
	{
		std::cerr << "error in SHAPE description" << std::endl;
		return false;
	}

	unsigned int length = pPtr2-pPtr;
	strncpy((char*)&(this->name), pPtr, length);	// extract the name
	name[length] = 0; // append terminating zero
	// now begin extracting model information, starting after '{'
	pPtr = strchr(pPtr2, '{');
	if ( pPtr == NULL )
	{
		std::cerr << "error in SHAPE description" << std::endl;
		return false;
	}
	pPtr = strchr(pPtr, ','); // skip the first number, it is always 0
	pPtr++; // skip the comma
	
	shape.numVertices = strtol(pPtr, &pPtr2, 10);
	pPtr = ++pPtr2; // again, skip comma
	shape.numPolys = strtol(pPtr, &pPtr2, 10);

	// let's get the vertices now
	pPtr = strstr(pData, "VECTOR"); // begins with "VECTOR xxxx_vertex[] = {"
	if ( pPtr == NULL )
	{
		std::cerr << "VECTOR description missing" << std::endl;
		return false;
	}
	pPtr = strchr(pPtr, '{'); // find opening brace
	if ( pPtr == NULL )
	{
		std::cerr << "error in VECTOR description" << std::endl;
		return false;
	}
	int i, j;
	shape.pVertices = new Vertex[shape.numVertices];
	for ( i=0; i < shape.numVertices; i++ )
	{// read all vertices
		pPtr++; // skip VECTOR opening brace or last vertex's closing brace
		pPtr = strchr(pPtr, '{'); // every vertex begins with a '{'
		pPtr++; // so skip it
		for ( j=0; j < 3; j++ )	// every vertex has 3 coordinates
		{
			shape.pVertices[i].v[j] = strtol(pPtr, &pPtr2, 10);
			pPtr = ++pPtr2;
		}
		shape.pVertices[i].v[1] = -shape.pVertices[i].v[1]; // mirror the y-coordinate
		if (pPtr[0] != '}')	// find this vertex's closing brace
			pPtr = strchr(pPtr, '}');
	}

	// and finally, the polygons
	pPtr = strstr(pData, "SPOLY"); // begins with "SPOLY xxxx_spoly[] = {"
	if ( pPtr == NULL )
	{
		std::cerr << "SPOLY description missing" << std::endl;
		return false;
	}
	pPtr = strchr(pPtr, '{'); // find opening brace
	if ( pPtr == NULL )
	{
		std::cerr << "error in SPOLY description" << std::endl;
		return false;
	}
	shape.pPolys = new NSHPoly[shape.numPolys];
	for ( i=0; i < shape.numPolys; i++ )
	{// all polygons
		pPtr++; // skip SPOLY opending brace or last poly's closing brace
		pPtr = strchr(pPtr, '{'); // every polygon start with that
		pPtr++; // and... skip
		shape.pPolys[i].colorindex = strtol(pPtr, &pPtr2, 10);
		pPtr = ++pPtr2; // skip comma
		shape.pPolys[i].numVertices = strtol(pPtr, &pPtr2, 10);
		pPtr = strchr(pPtr, '{'); // the vertex indices have extra braces
		pPtr++;
		for ( j=0; j < shape.pPolys[i].numVertices; j++ )
		{
			shape.pPolys[i].vertices[j] = strtol(pPtr, &pPtr2, 10);
			pPtr = ++pPtr2;
		}
		if (pPtr[0] != '}')	// find this polygon's vertex indices closing brace
			pPtr = strchr(pPtr, '}');
		pPtr++;
		if (pPtr[0] != '}')	// find this polygon's closing brace
			pPtr = strchr(pPtr, '}');
	}
	return true;
}

/** @brief Loads a NSH file to memory.
  *
  * After loading, the model will be parsed by a call to parseData().
  * Then, a call to calculateNormals() will create the normals for it.
  * @param filename The NSH model file name.
  * @return false if anything strange happened and the model could not
  *			be loaded or parsed properly. In this case, one should assume
  *			that there is no usable OpenGL display list afterwards.
  */
bool NSHModel::loadFromFile(const char *filename)
{
	FILE *pFile = fopen(filename, "rb"); // open for reading, binary to suppress CR/LF translation
	if (pFile == NULL)
	{
		std::cerr << "error: file could not be opened for reading\n" << std::endl;
		return false;
	}
	fseek(pFile, 0, SEEK_END);
	unsigned long	filesize = ftell(pFile),	// get the file size
					bytesread;
	fseek(pFile, 0, SEEK_SET); // set the pointer back to the beginning of the file
	char *pData = new char[filesize];
	if ( (bytesread=fread(pData, 1/*sizeof(unsigned char)*/, filesize, pFile)) < filesize )
	{
		std::cerr << "error: expected " << filesize << " bytes but read " << bytesread << std::endl;
		fclose(pFile);
		delete[] pData;
		return false;
	}
	fclose(pFile);
	bool ret = parseData(pData);	// now extract the model data
	delete[] pData;
	
	calculateNormals();

	return ret;
}

/** @brief Creates a display list from the model data.
  *
  * A model should have been loaded before.
  * @param list_number The OpenGL display list number. Use this later as a
  *			parameter to glCallList().
  * @param lighting Whether the model should be lighted.
  * @param smooth Whether smoothed surface normals should be used for
  *			lighting. Has no effect if #lighting is false.
  */
bool NSHModel::createDisplayList(const int list_number, bool lighting, bool smooth)
{
	glNewList(list_number, GL_COMPILE);
		draw(lighting, smooth);
	glEndList();
	return true;
}

/*void NSHModel::draw(bool lighting, bool smooth)
// draws the model, with lighting if desired, using smooth normals if desired
// this version tries to break all polygons down to triangles to avoid non-convex polygons
// may go wrong (depends on polygon orientation ?)
{
	const int QUADTRI1[] = {0, 1, 2};
	const int QUADTRI2[] = {0, 2, 3};
	const int PENTATRI3[]= {0, 4, 3};
	int i, j, index;
	Vertex *pvector;
	Vector3D *pnormal;

	glPushAttrib(GL_LIGHTING_BIT | GL_POLYGON_BIT);
	//glCullFace(GL_FRONT);	// NSH format seems to use CW polygons
	glDisable(GL_CULL_FACE);
	//glFrontFace(GL_CW);
	if (!lighting)
		glDisable(GL_LIGHTING);
	glBegin(GL_TRIANGLES);
		for ( i=0; i<shape.numPolys; i++ )
		{
			if ( shape.pPolys[i].numVertices > 2 )
			{
				glColor3dv(NSH_COLOR+(shape.pPolys[i].colorindex*3));
				pnormal = &(shape.pPolys[i].normal);
				for ( j=0; j<3; j++ )
				{
					index = shape.pPolys[i].vertices[QUADTRI1[j]];
					if ( smooth )
						pnormal = &(shape.pNormals[index]);
					glNormal3f(pnormal->x, pnormal->y, pnormal->z);
					pvector = &(shape.pVertices[index]);
					glVertex3f(pvector->v[0], pvector->v[1], pvector->v[2]);
				}
				if ( shape.pPolys[i].numVertices > 3 )
				{
					for ( j=0; j<3; j++ )
					{
						index = shape.pPolys[i].vertices[QUADTRI2[j]];
						glNormal3f(pnormal->x, pnormal->y, pnormal->z);
						if ( smooth )
							pnormal = &(shape.pNormals[index]);
						pvector = &(shape.pVertices[index]);
						glVertex3f(pvector->v[0], pvector->v[1], pvector->v[2]);
					}
				}
				if ( shape.pPolys[i].numVertices > 4 )
				{
					for ( j=0; j<3; j++ )
					{
						index = shape.pPolys[i].vertices[PENTATRI3[j]];
						if ( smooth )
							pnormal = &(shape.pNormals[index]);
						glNormal3f(pnormal->x, pnormal->y, pnormal->z);
						pvector = &(shape.pVertices[index]);
						glVertex3f(pvector->v[0], pvector->v[1], pvector->v[2]);
					}
				}
			}
		}
	glEnd();//triangles
	glBegin(GL_LINES);
		for ( i=0; i<shape.numPolys; i++ )
			if ( shape.pPolys[i].numVertices==2 )	// lines only
			{
				glColor3dv(NSH_COLOR+(shape.pPolys[i].colorindex*3));
				for ( j=0; j<2; j++ )
				{
					index = shape.pPolys[i].vertices[j];
					pvector = &(shape.pVertices[index]);
					glVertex3f(pvector->v[0], pvector->v[1], pvector->v[2]);
				}
			}
	glEnd();//lines
	glPopAttrib();
}*/

/** @brief Draw the model using OpenGL commands.
  *
  * This version draws the model by drawing all the polygons.
  * This may go wrong if a polygon is non-convex!
  * @param lighting Whether the model should be lit, that is, whether
  *			glDisable(GL_LIGHTING) should be placed in the display list.
  * @param smooth If true, the more or less accurately smoothed normal
  *			vectors (see calculateNormals()) will be used for lighting.
  */
void NSHModel::draw(bool lighting, bool smooth)
{
	int i, j, index;
	Vertex *pvector;
	Vector3D *pnormal;

	glPushAttrib(GL_LIGHTING_BIT | GL_POLYGON_BIT);
	//glCullFace(GL_FRONT);	// NSH format seems to use CW polygons
	glDisable(GL_CULL_FACE);
	//glFrontFace(GL_CW);
	if (!lighting)
		glDisable(GL_LIGHTING);
	for ( i=0; i<shape.numPolys; i++ )
	{
		if ( shape.pPolys[i].numVertices > 2 )
		{
			glColor3dv(NSH_COLOR+(shape.pPolys[i].colorindex*3));
			pnormal = &(shape.pPolys[i].normal);
			glBegin(GL_POLYGON);
				for ( j=0; j<shape.pPolys[i].numVertices; j++ )
				{
					index = shape.pPolys[i].vertices[j];
					if ( smooth )
						pnormal = &(shape.pNormals[index]);
					glNormal3f(pnormal->x, pnormal->y, pnormal->z);
					pvector = &(shape.pVertices[index]);
					glVertex3f(pvector->v[0], pvector->v[1], pvector->v[2]);
				}
			glEnd();
		}
	}
	glBegin(GL_LINES);
		for ( i=0; i<shape.numPolys; i++ )
			if ( shape.pPolys[i].numVertices==2 )	// lines only
			{
				glColor3dv(NSH_COLOR+(shape.pPolys[i].colorindex*3));
				for ( j=0; j<2; j++ )
				{
					index = shape.pPolys[i].vertices[j];
					pvector = &(shape.pVertices[index]);
					glVertex3f(pvector->v[0], pvector->v[1], pvector->v[2]);
				}
			}
	glEnd();//lines
	glPopAttrib();
}

/** @brief Draws the model wire frame without lighting.
  */
void NSHModel::drawWireFrame()
{
	int i, j, index;
	Vertex *pvertex;

	glPushAttrib(GL_LIGHTING_BIT);
	glDisable(GL_LIGHTING);
	glColor3f(1.0, 1.0, 1.0);
	for ( i=0; i<shape.numPolys; i++ )
	{
		if ( shape.pPolys[i].numVertices > 2 )
		{
			glBegin(GL_LINE_LOOP);
			for ( j=0; j<shape.pPolys[i].numVertices; j++ )
			{
				index = shape.pPolys[i].vertices[j];
				pvertex = &(shape.pVertices[index]);
				glVertex3f(pvertex->v[0], pvertex->v[1], pvertex->v[2]);
			}
			glEnd();
		}
		else
		{
			glBegin(GL_LINES);
			for ( j=0; j<2; j++ )
			{
				index = shape.pPolys[i].vertices[j];
				pvertex = &(shape.pVertices[index]);
				glVertex3f(pvertex->v[0], pvertex->v[1], pvertex->v[2]);
			}
			glEnd();
		}
	}
	glPopAttrib();
}

/** Calculates the surface normals for the model.
  * This is done on a per-polygon basis first.
  * Then, these normals will be used to calculate smoothed surface normals,
  * but in a very primitive way. So do not expect any miracles.
  */
void NSHModel::calculateNormals()
{
	int i, j, k, index;
	Vertex *pvector;

	for (i=0; i<shape.numPolys; i++)
	{// calculate a normal for every polygon
		if (shape.pPolys[i].numVertices < 3)
			continue;

		// We need three vertices of the polygon that do not lie in a straight line.
		// Straight line checking is not done now. TODO
		index = shape.pPolys[i].vertices[0];
		pvector = &(shape.pVertices[index]);
		Vector3D v1(pvector->v);

		index = shape.pPolys[i].vertices[1];
		pvector = &(shape.pVertices[index]);
		Vector3D v2(pvector->v);

		index = shape.pPolys[i].vertices[2];
		pvector = &(shape.pVertices[index]);
		Vector3D v3(pvector->v);
		// The normal is the cross product [v1-v2]x[v2-v3].
		shape.pPolys[i].normal = CrossProduct(v1-v2, v2-v3); // nsh clockwise polys
		shape.pPolys[i].normal.normalize();
	}

	// calculate smooth surface normals
	Vector3D avg;
	Vertex *pv1, *pv2;
	int polycount;
	shape.pNormals = new Vector3D[shape.numVertices];
	for (i=0; i<shape.numVertices; i++)
	{// calculate the smoothed normals for all vertices
		//std::cout << "processing vertex " << i << std::endl;
		avg.x = avg.y = avg.z = 0;
		polycount = 0;
		pv1 = &(shape.pVertices[i]);
		for (j=0; j<shape.numPolys; j++)
		{// check all polygons
			for (k=0; k<shape.pPolys[j].numVertices; k++)
			{
				index = shape.pPolys[j].vertices[k];
				pv2 = &(shape.pVertices[index]);
				//if ( shape.pPolys[j].vertices[k] == i ) // this way works with the nwdraw generated models
				// ...and this way also works with converted Anim8or models:
				if ( (pv1->v[0] == pv2->v[0]) && (pv1->v[1] == pv2->v[1]) && (pv1->v[2] == pv2->v[2]) )
				{// polygon j shares vertex i, so add it
					Vector3D v1(shape.pVertices[i].v); 
					float angle = fabs(DotProduct(v1, shape.pPolys[j].normal));
					//std::cout << "    adding normal, angle = " << angle << "degrees" << std::endl;
					if ( angle <= 90 )
					{
						avg = avg + shape.pPolys[j].normal;
						polycount++;
					}
				}
			}
		}
		avg = avg * (1.0/(float)polycount);
		avg.normalize();
		shape.pNormals[i] = Vector3D(avg);
	}
}

