/** @file nshmodel.h
  * @brief NSHModel declaration, NSH constants.
  */

#ifndef nsh_include
#define nsh_include

#include "vector3d.h"

const int MODEL_NAME_LENGTH = 200;

/** @brief Approximated NetWars colors.
  */
const GLdouble	NSH_COLOR[96] =	{	0.0,	0.0,	0.0,	// 0  black
									0.0,	0.0,	0.6,	// 1  dark blue
									0.0,	0.6,	0.0,	// 2  dark green
									0.0,	0.6,	0.6,	// 3  dark cyan
									0.6,	0.0,	0.0,	// 4  dark red
									0.6,	0.0,	0.6,	// 5  dark magenta
									0.6,	0.5,	0.2,	// 6  brown
									0.6,	0.6,	0.6,	// 7  grey

									0.3,	0.3,	0.3,	// 8  dark grey
									0.3,	0.3,	1.0,	// 9  light blue
									0.3,	1.0,	0.3,	// 10 light green
									0.3,	1.0,	1.0,	// 11 light cyan
									1.0,	0.3,	0.3,	// 12 light red
									1.0,	0.3,	1.0,	// 13 light magenta
									1.0,	1.0,	0.3,	// 14 light yellow
									1.0,	1.0,	1.0,	// 15 white

									0.0,	0.0,	0.0,	// 16 black
									0.0,	0.0,	0.4,	// 17 dark blue
									0.0,	0.4,	0.0,	// 18 dark green
									0.0,	0.4,	0.4,	// 19 dark cyan
									0.4,	0.0,	0.0,	// 20 dark red
									0.4,	0.0,	0.4,	// 21 dark magenta
									0.4,	0.3,	0.1,	// 22 brown
									0.4,	0.4,	0.4,	// 23 grey

									0.2,	0.2,	0.2,	// 24 dark grey
									0.2,	0.2,	0.8,	// 25 light blue
									0.2,	0.8,	0.2,	// 26 light green
									0.2,	0.8,	0.8,	// 27 light cyan
									0.8,	0.2,	0.2,	// 28 light red
									0.8,	0.2,	0.8,	// 29 light magenta
									0.8,	0.8,	0.2,	// 30 light yellow
									0.8,	0.8,	0.8,	// 31 white
								};

/** @brief A simple 3d vector for NSH models.
  */
struct Vertex
{
	short		v[3];
};

/** @brief a polygon description as defined in .nsh files.
  */
struct NSHPoly
{
	unsigned char	numVertices;	///< NSH polygons may have 2 to 5 points
	unsigned char	colorindex;		///< index into the 16/32 color table
	short			vertices[5];	///< indices into the vertex table of this model
	Vector3D		normal;			///< the surface normal (not in file, calculation)
};

/** @brief a Shape description as defined in .nsh files.
  */
struct NSHShape
{
	int				reserved,		///< always 0
					numVertices,	///< number of vertices in this model
					numPolys;		///< number of polygons in this model
	Vertex			*pVertices;		///< all vertices of this model
	Vector3D		*pNormals;		///< the smoothed (averaged) normals for the respective vertices
	NSHPoly			*pPolys;
};

/** @brief NSH model loading, parsing and management.
  *
  * NSH models are stored in .nsh files. These are actually C headers that
  * can be directly included in C/C++ programs. This implies, however, that
  * we always want to use these same models. To avoid that, this class
  * provides a simple (and very likely, buggy and sensitive to .nsh
  * correctness) parser.
  *
  * One object of this class should be used for every NSH model you want to
  * handle. The object should first be loaded from a file. Then an OpenGL
  * display list can be created.
  */
class NSHModel
{
public:
	/*constructor*/	NSHModel();
	/*destructor*/	~NSHModel();
	bool			loadFromFile(const char *filename);
	bool			createDisplayList(const int list_number, bool lighting, bool smooth);
	void			draw(bool lighting, bool smooth);
	void			drawWireFrame();
	inline char*	getName() { return name; } ///< @returns The model name.

private:	
	bool		parseData(const char *pData);
	void		calculateNormals();

	/// The model name as read from the file.
	char		name[MODEL_NAME_LENGTH];

	/// Holds the model data after parsing.
	NSHShape	shape;
};

#endif

