#include <string.h>
#include <stdlib.h>
#include "particlesystem.h"
#include "stdinclu.h"

/** @brief Initial creation of the particles.
  *
  * All particles start at the center of the particle system, having a random
  * direction and random velocity.
  */
void ParticleSystem::createParticles()
{
	Vector3D direction;
	GLdouble	red[3] = {1.0, 0.0, 0.0},
				yellow[3] = {1.0, 1.0, 0.0};
	for ( int i=0; i<nParticles; i++ )
	{
		direction = Vector3D((rand()%201-100)/100.0f, (rand()%201-100)/100.0f, (rand()%201-100)/100.0f);
		direction.normalize();
		Particle particle(position, direction, (200+rand()%201-100)/100.0f, -0.00066, i%2==0? red : yellow);
		particles.push_back(particle);
	}
}

/** @brief Creates a particle system.
  *
  * The system will have the standard number of particles and the center
  * position (0, 0, 0).
  */
ParticleSystem::ParticleSystem()
{
	position = Vector3D(0, 0, 0);
	nParticles = MAX_PARTICLES;
	createParticles();
}

/** @brief Creates a specific particle system.
  *
  * @param vpos The center of the particle system.
  * @param nParticleCount The number of particles of this system.
  */
ParticleSystem::ParticleSystem(const Vector3D& vpos, int nParticleCount)
{
	position = vpos;
	nParticles = nParticleCount < MAX_PARTICLES ? nParticleCount : MAX_PARTICLES;
	createParticles();
}

ParticleSystem::~ParticleSystem()
{
}

/** @brief Calculates the time steps for all particles.
  *
  * Every particle that is too old is deleted from the particle system.
  *
  * @param timediff Time difference since last step.
  * @returns false if all particles are dead.
  */
bool ParticleSystem::move(int timediff)
{
	typedef ParticleList::iterator iterator;
	for ( iterator it = particles.begin(); it!=particles.end(); ++it)
	{
		if ( it->velocity <= PARTICLE_MIN_VELOCITY )
		{// do some tricks because erasing invalidates the iterator
			iterator del = it;
			--it;
			particles.erase(del);
		}
		else
			it->move(timediff);
	}

	return ( !(particles.empty()) );
}

/** @brief Draws all particles.
  */
void ParticleSystem::draw()
{
	for ( ParticleList::iterator it = particles.begin(); it!=particles.end(); ++it)
	{
		glPushMatrix();
			glColor3dv(it->color);
			glTranslatef(it->position.x, it->position.y, it->position.z);
			glCallList(DL_PARTICLE);
		glPopMatrix();
	}
}
