/** @file pirateai.cpp
  * @brief PirateAI implementation.
  */

#include <cstdlib>
#include <cmath>
#include "stdinclu.h"
#include "constants.h"
#include "shipai.h"
#include "spaceobject.h"
#include "podai.h"
#include "pirateai.h"
#include "game.h"

PirateAI::PirateAI(SpaceObject* pprnt)
{
	pParent = pprnt;			// keep track of my parent space object
	pParent->collectedMissiles(); // TODO can we handle this some other way?
	pParent->setShield(PIRATE_SHIELD_STRENGTH);
	state = ATTACK_PLAYER;		// pirate first tries to launch a missile
	pTarget = pPlayer = NULL;	// aim for player
	pTargetPod = NULL;
	pGame = NULL;
	maxvelocity = PIRATE_MAX_VELOCITY;
	velocity = 100;
	turnspeed = PIRATE_TURN_SPEED;
}

void PirateAI::step(int timediff)
{
	static Vector3D		distance;	// between me and my target
	Vector3D			tmpv;
	switch (state)
	{
		case ATTACK_PLAYER:	// try to shoot a missile at target
			trackTarget(timediff);	// turn and move towards target
			distance = pTarget->getPosition() - pParent->getPosition();
			if (distance.getLength() <= PIRATE_MISSILE_DISTANCE)
			{// launch missile and switch state
				pParent->launchMissile();
				setTargetPosition((pParent->getPosition()-pTarget->getPosition())*10000);
				state = FLEE;
			}
			break;
		case FLEE: // pTarget should still be the PLAYER, try to get some distance
			setTargetPosition((pParent->getPosition()-pTarget->getPosition())*10000);
			moveToTarget(targetPos, timediff);
			distance = pTarget->getPosition() - pParent->getPosition();
			if (distance.getLength() > PIRATE_FLEE_DISTANCE)
			{// aim at a pod and switch state
				pTargetPod = pGame->getFreePod();
				if ( pTargetPod == NULL )
				{// if there is no pod to be hijacked, attack the player again
					pTarget = pPlayer;
					state = ATTACK_PLAYER;
				}
				else
				{
					pTarget = pTargetPod;
					pTargetAI = (PodAI*)pTargetPod->getAI();
					state = APPROACH_POD;
				}
			}
			break;
		case APPROACH_POD:	// slow down and approach pod until within docking range
			watchPod();
			if ( state != APPROACH_POD ) // watchPod() may have changed the state
				break;
			trackTarget(timediff);
			distance = pTargetPod->getPosition() - pParent->getPosition();
			if ( distance.getLength() < PIRATE_BRAKE_DISTANCE )
			{// fly slower in vicinity of pod
				velocity = PIRATE_APPROACH_VELOCITY;
				if ( distance.getLength() < PIRATE_DOCK_DISTANCE )
				{// stop and switch state
					velocity = 0;
					setTargetPosition((pParent->getPosition()-pTarget->getPosition())*20000);
					pTargetAI->dock(pParent);
					state = DOCK_POD;
				}
			}
			break;
		case DOCK_POD:	// just turns the pirate to fleeing direction
			moveToTarget(targetPos, timediff);
			tmpv = targetPos - pParent->getPosition();
			tmpv.normalize();
			if ( DEGREES(acos(DotProduct(pParent->getLookDir(), tmpv))) <= 1)
			{
				pTargetAI->hijack();
				velocity = 100;
				pTarget = pPlayer;	// get destruction distance to player
				state = FLEE_POD;
				fleeingTime = 0;
			}
			break;
		case FLEE_POD:	// move until destruction distance is reached
			moveToTarget(targetPos, timediff);
			distance = pTarget->getPosition() - pParent->getPosition();
			fleeingTime += timediff;
			if (    fleeingTime          > PIRATE_DESTRUCTION_TIME 
				 && distance.getLength() > PIRATE_DESTRUCTION_DISTANCE )
			{
				pParent->setSelfDestruction();
			}
			break;
	}
}

/**
  * Keep looking at the targeted pod to see if another ship attacks it.
  */
void PirateAI::watchPod()
{
	if ( pTargetAI->isBeingHijacked() )
	{// current target pod invalid, get another one
		if ( (pTargetPod = pGame->getFreePod()) == NULL )
		{// no pod available, flee and attack again
			pTarget = pPlayer;
			state = FLEE;
		}
		else
		{
			pTarget = pTargetPod;
			pTargetAI = (PodAI*)pTargetPod->getAI();
		}
	}
}

/**
  * Call this to notify pirates about a pod being destroyed.
  * The pirate checks if it was his target pod and will react properly.
  */
void PirateAI::notifyPodDestruction(SpaceObject* pobj)
{
	if ( pobj == pTargetPod )
	{// it was my pod, flee from player, then get another one
		pTargetPod = NULL;
		pTarget = pPlayer;
		velocity = 100;
		state = FLEE;
	}
}

