#include <stdlib.h>
#include <cassert>
#include "stdinclu.h"
#include "constants.h"
#include "podai.h"
#include "pirateai.h"
#include "fighterai.h"
#include "guardianai.h"
#include "commanderai.h"
#include "spaceobject.h"
#include "bullet.h"
#include "timing.h"
//#include "nwsound.h"

// forward declarations
//extern NWSound* pSound;

void SpaceObject::initMembers()
{
	lookdir.x = lookdir.y = 0; lookdir.z = 1;
	right.x = 1; right.y = right.z = 0;
	up.x = 0; up.y = 1; up.z = 0;
	flydir = lookdir;
	
/*	rotation.m[ 0] = right.x;	rotation.m[ 1] = right.y;	rotation.m[ 2] = right.z;
	rotation.m[ 4] = up.x;		rotation.m[ 5] = up.y;		rotation.m[ 6] = up.z;
	rotation.m[ 8] = lookdir.x;	rotation.m[ 9] = lookdir.y;	rotation.m[10] = lookdir.z;
	rotation.m[15] = 1;
*/	
	rotation.m[ 0] = right.x;	rotation.m[ 4] = right.y;	rotation.m[ 8] = right.z;
	rotation.m[ 1] = up.x;		rotation.m[ 5] = up.y;		rotation.m[ 9] = up.z;
	rotation.m[ 2] = lookdir.x;	rotation.m[ 6] = lookdir.y;	rotation.m[10] = lookdir.z;
	rotation.m[15] = 1;

	autorot[0] = autorot[1] = autorot[2] = 0;
	cloaked = false;
	collision = false;
	bAI = false;
	//pAI = NULL;
	pLockTarget = 0;
	velocity = 0;
	slidevel = 0;
	maxvelocity = MAX_VELOCITY;
	lastshot = 0;
	lastaccel = 0;
	exFirePower = 0;
	exThrust = 0;
	exCloak = 0;
	missiles = 16;
	missileflag = false;
	flags = 0;
	lastHitTC = 0;
	lastmissile = 0;
	// Create all bullets in advance to avoid allocation/deallocation slowdowns.
	// All bullets are at (0, 0, 0) and not active.
	for ( int i=0; i<MAX_BULLETS; i++ )
		m_pbullets[i] = new Bullet();
}

SpaceObject::SpaceObject(int iModel, int iShield/*=MAX_SHIELD_CAPACITY*/)
{
	model = iModel;
	shield = iShield;
	position.x = 0;
	position.y = 0;
	position.z = 0;

	initMembers();
}

SpaceObject::SpaceObject(int iModel, GLfloat xpos, GLfloat ypos, GLfloat zpos, int iShield/*=1*/)
{
	model = iModel;
	shield = iShield;
	position.x = xpos;
	position.y = ypos;
	position.z = zpos;

	initMembers();
}

SpaceObject::~SpaceObject()
{
	for ( int i=0; i<MAX_BULLETS; i++ )
		if ( m_pbullets[i] )
		{
			delete m_pbullets[i];
			m_pbullets[i] = 0;
		}
	if ( bAI==true && pAI!=NULL )
	{
		delete pAI;
	}
}

/** @brief Moves the object and every object belonging to it.
  *
  * Movement is the automatic rotation, collision rotation and translation
  * in the world.
  * @param timediff Time difference to last movement.
  * @returns false if this object wants to be destroyed.
  */
bool SpaceObject::move(int timediff)
{
	if ( bAI==true && pAI!=NULL )
		pAI->step(timediff);

	if ( collision )
	{
		rotateX( collrot[0]*(float)timediff );
		rotateY( collrot[1]*(float)timediff );
		rotateZ( collrot[2]*(float)timediff );
		position = position + (colldir*((maxvelocity*velocity*0.01)*timediff));
		collrot[0] -= (COLLISION_DECREASE * timediff) * (collrot[0]>0 ? 1 : -1);
		collrot[1] -= (COLLISION_DECREASE * timediff) * (collrot[1]>0 ? 1 : -1);
		collrot[2] -= (COLLISION_DECREASE * timediff) * (collrot[2]>0 ? 1 : -1);
		if ( collrot[2] <= 0 )
			collision = false;
	}
	else
	{
		// auto rotation
		rotateX( autorot[0]*(float)timediff );
		rotateY( autorot[1]*(float)timediff );
		rotateZ( autorot[2]*(float)timediff );

		// translation
		flydir = flydir*3.0 + lookdir; // TODO check that constant multiplier (inertia!)
		flydir.normalize();
		position = position + (flydir*((maxvelocity*velocity*0.01)*timediff));
		//position = position + (lookdir*((maxvelocity*velocity*0.01)*timediff));
		position = position + (right*(slidevel*timediff));
	}
	
	int i;
	for ( i=0; i<MAX_BULLETS; i++ )
	{
		if ( (m_pbullets[i]->isActive())==true )
		{
			if ( (m_pbullets[i]->move(timediff)) == false )
			{// too old
				m_pbullets[i]->stop();
			}
		}
	}

	if ( cloaked )
	{	
		exCloak -= timediff * CLOAK_DRAIN_SPEED;
		if ( exCloak <= 0 )
			cloaked = false;
	}

	if ( exThrust>0 && velocity!=0)
	{
		exThrust -= timediff * THRUST_DRAIN_SPEED;
		if ( exThrust <= 0 )
			maxvelocity = MAX_VELOCITY;
	}
	return ( (flags & SOB_SELF_DESTRUCTION) != SOB_SELF_DESTRUCTION );
}

void SpaceObject::draw()
{
	if ( cloaked )
		return;

	float		m[16];

	glTranslated(position.x, position.y, position.z);

	m[ 0] = right.x; m[ 4] = up.x; m[ 8] = lookdir.x; m[12] = 0.0f;
	m[ 1] = right.y; m[ 5] = up.y; m[ 9] = lookdir.y; m[13] = 0.0f;
	m[ 2] = right.z; m[ 6] = up.z; m[10] = lookdir.z; m[14] = 0.0f;
	m[ 3] = 0;       m[ 7] = 0;    m[11] = 0;         m[15] = 1.0f;
	glMultMatrixf(m);

	glCallList(model);
}

void SpaceObject::drawBullets()
{
	Vector3D		pos;

	glColor3f(1.0f, 1.0f, 1.0f);
	for ( int i=0; i<MAX_BULLETS; i++ )
		if ( (m_pbullets[i]->isActive())==true )
		{
			glPushMatrix();
				pos = m_pbullets[i]->getPosition();
				glTranslatef(pos.x, pos.y, pos.z);
				glCallList(m_pbullets[i]->getModel());
			glPopMatrix();
		}
}

void SpaceObject::setCamera()
{
	float m[16];
	m[ 0] = right.x; m[ 1] = up.x; m[ 2] = -lookdir.x; m[ 3] = 0.0f;
	m[ 4] = right.y; m[ 5] = up.y; m[ 6] = -lookdir.y; m[ 7] = 0.0f;
	m[ 8] = right.z; m[ 9] = up.z; m[10] = -lookdir.z; m[11] = 0.0f;
	m[12] = 0;		 m[13] = 0;	   m[14] = 0;		   m[15] = 1.0f;

	glLoadMatrixf(m);

	glTranslated(-position.x, -position.y, -position.z);
}

void SpaceObject::cameraRotate()
{
	float m[16];
	m[ 0] = right.x; m[ 1] = up.x; m[ 2] = -lookdir.x; m[ 3] = 0.0f;
	m[ 4] = right.y; m[ 5] = up.y; m[ 6] = -lookdir.y; m[ 7] = 0.0f;
	m[ 8] = right.z; m[ 9] = up.z; m[10] = -lookdir.z; m[11] = 0.0f;
	m[12] = 0;		 m[13] = 0;	   m[14] = 0;		   m[15] = 1.0f;

	glLoadMatrixf(m);
}

void SpaceObject::cameraTranslate()
{
	if ( exThrust > 0 && velocity>80 )
		glTranslated((rand()%100-50)/100, (rand()%100-50)/100, (rand()%100-50)/100);
	glTranslated(-position.x, -position.y, -position.z);
}

/// Not needed!?
/*
void SpaceObject::setLookDir(GLfloat x, GLfloat y, GLfloat z)
{
	Vector3D v(x, y, z);
	// TODO: set right and up accordingly
	v.normalize();
	lookdir = v;
}
*/

/// Not needed!?
/*
void SpaceObject::setLookDir(const Vector3D& v)
{
	lookdir = v;
	// TODO: set right and up accordingly
	lookdir.normalize();
}
*/

void SpaceObject::setOrientationVectors(const Vector3D& nlookdir, const Vector3D& nright, const Vector3D& nup)
{
	lookdir = nlookdir;
	right = nright;
	up = nup;
}

void SpaceObject::rotateX(GLfloat amount)
{
	if ( amount==0 )
		return;
	glPushMatrix();
		glLoadMatrixf(rotation.m);
		glRotatef(amount, right.x, right.y, right.z);
		glGetFloatv(GL_MODELVIEW_MATRIX, rotation.m);
		glFlush();

		right.x = rotation.m[0];right.y = rotation.m[4];right.z=rotation.m[8];
		up.x = rotation.m[1];	up.y = rotation.m[5];	up.z = rotation.m[9];
		lookdir.x = rotation.m[2]; lookdir.y = rotation.m[6]; lookdir.z = rotation.m[10];
		
		right.normalize();
		up.normalize();
		lookdir.normalize();

	glPopMatrix();

/*	amount = RADIANS(amount);
	GLfloat		ca = cos(amount),
				sa = sin(amount);

	Matrix4x4	rotX;
	rotX.m[ 0] = 1;	rotX.m[ 1] = 0;		rotX.m[ 2] = 0;		rotX.m[ 3] = 0;
	rotX.m[ 4] = 0;	rotX.m[ 5] = ca;	rotX.m[ 6] = -sa;	rotX.m[ 7] = 0;
	rotX.m[ 8] = 0;	rotX.m[ 9] = sa;	rotX.m[10] = ca;	rotX.m[11] = 0;
	rotX.m[12] = 0;	rotX.m[13] = 0;		rotX.m[14] = 0;		rotX.m[15] = 1;

	rotation = rotation * rotX;

	right.x = rotation.m[0];right.y = rotation.m[4];right.z=rotation.m[8];
	up.x = rotation.m[1];	up.y = rotation.m[5];	up.z = rotation.m[9];
	lookdir.x = rotation.m[2]; lookdir.y = rotation.m[6]; lookdir.z = rotation.m[10];

	up.normalize();
	lookdir.normalize();
	right.normalize();*/
}

void SpaceObject::rotateY(GLfloat amount)
{
	if ( amount==0 )
		return;

	glPushMatrix();
		glLoadMatrixf(rotation.m);
		glRotatef(amount, up.x, up.y, up.z);
		glGetFloatv(GL_MODELVIEW_MATRIX, rotation.m);
		glFlush();

		right.x = rotation.m[0];right.y = rotation.m[4];right.z=rotation.m[8];
		up.x = rotation.m[1];	up.y = rotation.m[5];	up.z = rotation.m[9];
		lookdir.x = rotation.m[2]; lookdir.y = rotation.m[6]; lookdir.z = rotation.m[10];
		
		right.normalize();
		up.normalize();
		lookdir.normalize();

	glPopMatrix();

/*	amount = RADIANS(amount);
	GLfloat		ca = cos(amount),
				sa = sin(amount);

	Matrix4x4	rotY;
	rotY.m[ 0] = ca;	rotY.m[ 1] = 0;	rotY.m[ 2] = sa;	rotY.m[ 3] = 0;
	rotY.m[ 4] = 0;		rotY.m[ 5] = 1; rotY.m[ 6] = 0;		rotY.m[ 7] = 0;
	rotY.m[ 8] = -sa;	rotY.m[ 9] = 0; rotY.m[10] = ca;	rotY.m[11] = 0;
	rotY.m[12] = 0;		rotY.m[13] = 0; rotY.m[14] = 0;		rotY.m[15] = 1;

	rotation = rotation * rotY;

	right.x = rotation.m[0];right.y = rotation.m[4];right.z=rotation.m[8];
	up.x = rotation.m[1];	up.y = rotation.m[5];	up.z = rotation.m[9];
	lookdir.x = rotation.m[2]; lookdir.y = rotation.m[6]; lookdir.z = rotation.m[10];

	lookdir.normalize();
	right.normalize();
	up.normalize();*/
}

void SpaceObject::rotateZ(GLfloat amount)
{
	if ( amount==0 )
		return;

	glPushMatrix();
		glLoadMatrixf(rotation.m);
		glRotatef(amount, lookdir.x, lookdir.y, lookdir.z);
		glGetFloatv(GL_MODELVIEW_MATRIX, rotation.m);
		glFlush();

		right.x = rotation.m[0];right.y = rotation.m[4];right.z=rotation.m[8];
		up.x = rotation.m[1];	up.y = rotation.m[5];	up.z = rotation.m[9];
		lookdir.x = rotation.m[2]; lookdir.y = rotation.m[6]; lookdir.z = rotation.m[10];
		
		right.normalize();
		up.normalize();
		lookdir.normalize();

	glPopMatrix();

/*	amount = RADIANS(amount);
	GLfloat		ca = cos(amount),
				sa = sin(amount);

	Matrix4x4	rotZ;
	rotZ.m[ 0] = ca;	rotZ.m[ 1] = -sa;	rotZ.m[ 2] = 0;		rotZ.m[ 3] = 0;
	rotZ.m[ 4] = sa;	rotZ.m[ 5] = ca;	rotZ.m[ 6] = 0;		rotZ.m[ 7] = 0;
	rotZ.m[ 8] = 0;		rotZ.m[ 9] = 0;		rotZ.m[10] = 1;		rotZ.m[11] = 0;
	rotZ.m[12] = 0;		rotZ.m[13] = 0;		rotZ.m[14] = 0;		rotZ.m[15] = 1;

	rotation = rotation * rotZ;

	right.x = rotation.m[0];right.y = rotation.m[4];right.z=rotation.m[8];
	up.x = rotation.m[1];	up.y = rotation.m[5];	up.z = rotation.m[9];
	lookdir.x = rotation.m[2]; lookdir.y = rotation.m[6]; lookdir.z = rotation.m[10];

	lookdir.normalize();
	up.normalize();
	right.normalize();*/
}

/** @brief Shoots a bullet.
  *
  * This will happen with a specific frequency only, and only if the object
  * is not cloaked (see isCloaked()).
  *
  * @return true if a shot was fired, false if not.
  */
bool SpaceObject::shoot()
{
	long count = getElapsedTime();
	
	if ( (count - lastshot) < bulletinterval )
		return false;

	if ( cloaked )
		return false;
	
	for ( int i=0; i<MAX_BULLETS; i++ )
		if ( !(m_pbullets[i]->isActive()) )
		{
			m_pbullets[i]->start(position+(lookdir*10*BULLET_VELOCITY)-(up*30.0f), lookdir, exFirePower>0? DL_BULLET2 : DL_BULLET1);
			if ( exFirePower )
				exFirePower--;
			break;
		}
	lastshot = count;
	return true;
}

void SpaceObject::launchMissile()
{
	long count = getElapsedTime();
	
	if ( cloaked || missiles <= 0 )
		return;
	
	if ( (count - lastmissile) < MISSILE_INTERVAL )
		return;

	missileflag = true;
}

/** @brief Call this if the object has been hit by a bullet.
  *
  * By calling this method, the objects shields are decreased by the specified
  * amount.
  * @param power The bullet power. Shields are decreased by this value.
  * returns false if the object has no more shields left after this hit. It
  * 		should then be destroyed.
  */
bool SpaceObject::hitByBullet(int power)
{
	shield -= power;
	lastHitTC = getElapsedTime();
	return (shield > 0);
}

void SpaceObject::accelerate(int percentage, long tickcount)
{
	if ( tickcount-lastaccel < ACCEL_INTERVAL )
		return;

	velocity += percentage;
	if ( velocity > 100 )
		velocity = 100;
	if ( velocity < -100 )
		velocity = -100;
	lastaccel = tickcount;
}

void SpaceObject::cloakToggle()
{
	if ( cloaked )
	{
		cloaked = false;
		return;
	}
	if ( exCloak > 0 )
		cloaked = true;
}

/** @brief Initiates a ship-ship collision.
  *
  * Some reaction to a collision of this object with another (no weapon hit).
  * Sets the flag @c collision.
  * @param vector The impact vector.
  */
void SpaceObject::collide(const Vector3D& vector)
{
	if (!collision)
	{
		collision = true;
		colldir = vector;
		collrot[0] = vector.x *velocity*0.01;
		collrot[1] = vector.y *velocity*0.01;
		collrot[2] = 1 *velocity*0.01;
	}
}

/** @brief Adds a controlling AI to this space object.
  *
  * The object will then be controlled by the AI.
  * @param type The AI type to be used for this object. See AIShipType for
  * 		valid types. If NO_AI is
  *			specified, the current AI will be destroyed (not the object).
  */
void SpaceObject::setAI(AIShipType type)
{
	if (type == NO_AI)
	{
		bAI = false;
		if ( pAI!=NULL )
			delete pAI;
		pAI = NULL;
	}
	else
	{
		bAI = true;
		switch (type)
		{
			case POD_AI: pAI = new PodAI(this); break;
			case PIRATE: pAI = new PirateAI(this); break;
			case FIGHTER: pAI = new FighterAI(this); break;
			case GUARDIAN: pAI = new GuardianAI(this); break;
			case COMMANDER: pAI = new CommanderAI(this); break;
			default: assert(false);	// should never happen
		}
	}
}

