/** @file spaceobject.h
  * @brief SpaceObject declaration.
  */

#ifndef spaceobject_include
#define spaceobject_include

#include "matrix4x4.h"
#include "vector3d.h"
#include "bullet.h"
#include "shipai.h"
#include "timing.h"
#include "stdinclu.h"

// space object constants
#define MAX_SHIELD_CAPACITY	200
#define ACCEL_INTERVAL		50
#define COLLISION_DECREASE	0.0005
#define HIT_INDICATOR_TC	50

#define SOB_SELF_DESTRUCTION	(1 << 0)
#define SOB_SHOOT_MISSILE		(1 << 1)

/**	@brief Base class for all things that move around or just sit there.

	All velocities in units per millisecond.
	Rotation velocities in degrees per millisecond.
	And so on.
*/
class SpaceObject
{
public:
	/*constructor*/		SpaceObject(int iModel, int iShield = MAX_SHIELD_CAPACITY);
	/*constructor*/		SpaceObject(int iModel, GLfloat xpos, GLfloat ypos, GLfloat zpos, int iShield=1);
	/*destructor*/		~SpaceObject();
	bool				move(int timediff);
	void				draw();
	void				drawBullets();
	bool				shoot();
	void				launchMissile();
	void				collide(const Vector3D& vector);
	void				setAI(AIShipType);
	inline void			setModel(int inew) { model = inew; }
	inline int			getModel() const { return model; }
	inline Vector3D		getPosition() const { return position; }
	inline Vector3D		getLookDir() const { return lookdir; }
	inline Vector3D		getRight() const { return right; }
	inline Vector3D		getUp() const { return up; }
//	inline Bullet*		getPBullets() const { return pbullets; }
	inline void			setAutoRotation(GLfloat xr, GLfloat yr, GLfloat zr) 
							{ autorot[0]=xr; autorot[1]=yr; autorot[2]=zr; }
	inline void			setPosition(GLfloat px, GLfloat py, GLfloat pz)
							{ position.x=px; position.y=py; position.z=pz; }
	inline void			setPosition(const Vector3D& v) { position = v; }
	void				cloakToggle();
	inline void			unCloak() { cloaked = false; }
	inline void			collectedShield() { shield = MAX_SHIELD_CAPACITY; }
	inline void			collectedFirePower() { exFirePower += 100; }
	inline void			collectedThrust() { exThrust += 100; maxvelocity = MAX_VELOCITY*THRUST_FACTOR; }
	inline void			collectedMissiles() { missiles += 3; }
	inline void			collectedCloak() { exCloak += 100; }
	inline bool			isCloaked()			const { return cloaked; }
	inline bool			isAI()				const { return bAI; }
	inline ShipAI*		getAI()				const { return pAI; }
	inline void			setBulletInterval(int i) { bulletinterval = i; }
	inline void			setSelfDestruction() { flags |= SOB_SELF_DESTRUCTION; }
	inline bool			isSelfDestruction()	const { return (flags & SOB_SELF_DESTRUCTION == SOB_SELF_DESTRUCTION); }
	inline bool			isHit() { return (getElapsedTime() - lastHitTC)<HIT_INDICATOR_TC; }
	inline int			getVelocityPercent() const { return velocity; }
	inline int			getShield()			const { return shield; }
	inline void			setShield(int newshield) { shield = newshield; }
	inline int			getFirePower()		const { return exFirePower; }
	inline float		getCloak()			const { return exCloak; }
	inline float		getThrust()			const { return exThrust; }
	inline int			getMissileCount()	const { return missiles; }
	inline GLfloat		getVelocity()		const { return maxvelocity*velocity*0.01; }
	inline GLfloat		getMaxVelocity()	const { return maxvelocity; }
	inline void			setVelocity(int vel) { velocity = vel; }
	inline void			setSlideVel(GLfloat vel) { slidevel = vel; }
	inline bool			getMissileFlag()	const { return missileflag; }
	inline void			clearMissileFlag() { missileflag = false; missiles--; lastmissile = getElapsedTime(); }
	inline void			setLockTarget(SpaceObject* pobj) { pLockTarget = pobj; }
	inline SpaceObject*	getLockTarget() const { return pLockTarget; }
	//void				setLookDir(GLfloat x, GLfloat y, GLfloat z);
	//void				setLookDir(const Vector3D& v);
	void				setOrientationVectors(const Vector3D& nlookdir, const Vector3D& nright, const Vector3D& nup);
	void				setCamera();
	void				cameraRotate();
	void				cameraTranslate();
	void				rotateX(GLfloat angle);
	void				rotateY(GLfloat angle);
	void				rotateZ(GLfloat angle);
	bool				hitByBullet(int power);	// returns false if shield<=0
	void				accelerate(int percentage, long tickcount);

	Bullet*				m_pbullets[MAX_BULLETS];///< TODO: PLEASE change that (get getPBullets() to work)
protected:
	void				initMembers();
	Vector3D			position,	///< position in space
						flydir,		///< direction of object's movement
						lookdir,	///< looking direction
						right,		///< perpendicular to lookdir
						up,			///< perpendicular to lookdir
						colldir;	///< collision 'direction'
	GLfloat				maxvelocity,
						slidevel,	///< for moving left and right (positive values = right)
						autorot[3],	///< automatic rotation in every Move() (about x, y, z axis)
						collrot[3];	///< collision rotation
	long				lastshot,	///< Tick count of last bullet shot.
						lastmissile,///< Tick count of last missile launched.
						lastaccel;	///< Tick count of last acceleration.
	int					flags;		///< status flags
	int					velocity,	///< in % of maxvelocity. Keep in -100..+100!
						model,		///< 3d model #
						exFirePower,///< extra fire power
						shield,		///< amount of shield against bullets
						missiles,	///< missile count
						lastHitTC;	///< tick count of last bullet/missile impact
	int					bulletinterval;	///< Minimum time between bullets.
	float				exThrust,	///< extra thrust
						exCloak;	///< cloaking device capacity
	Matrix4x4			rotation;	///< like a copy of lookdir, right and up vectors
	bool				cloaked,	///< if true, object is invisible
						collision,	///< if true, object moves squiggles and turns over
						missileflag,///< true indicates that this object wants to launch a missile
						bAI;		///< true if this is an object (a ship) driven by an AI
	ShipAI*				pAI;
	SpaceObject*		pLockTarget;
};

#endif
